/*---------------------------------------------------------------------------*\

    FILE....: ECHO.CPP
    TYPE....: C++ Console program
    AUTHOR..: David Rowe
    DATE....: 6/12/00

    This program is used for testing Voicetronix card with Japan DID/BIN. 
    It works like this:

    1. It waits for (CAR) then thakes the line off-hook for DID/BIN collection.

    2. Once collected it hangs up (on-hook).
    
    3. If valid data is detected it is displayed. 

    4. It waits for two standard rings.
    
    5. It then takes the channel off hook and records for 4 seconds.  You can
       also stop recording by pressing any digit.
       
    6. It then plays back the recorded speech and hangs up.
    
    7. Press return to finish, otherwise it waits for the next ring (step 1)

    Compile: make jpecho
    Run....: ./jpecho 

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "../../src/vpbapi.h"
#include "../../src/verbose.h"
#include "../kbhit.h"
#include "../threads.h"

#define	N	(8000*4)

int ctwaitforevent(int h, int event_type, int timeout_ms);
void *port_thread(void *pv);
int arg_exists(int argc, char *argv[], char *arg);

int             threads_active; // the number of active threads.
pthread_mutex_t mutex;          // mutex to protect shared data
int             finito;         // flag to signal the program is finished.
float           gain;           // option software record gain
int		did_debug =0;

int main(int argc, char * argv[])
{
	int		arg;
	int	        num_ch,*h,i,ports_per_card;
	VPB_RECORD	record;
	pthread_t       *aport_thread;

	//	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);
	record.term_digits = "1234567890*#";  // any digit terminates record
	record.time_out = 4000;               // record finishes after 4 seconds
	pthread_mutex_init(&mutex,NULL);
	finito = 0;

	// Read command line switches	
       	if ((arg = arg_exists(argc, argv, "--debug")) != 0) 
	{
		did_debug=1;
		printf("Debug files enabled\n");
	}				 
       	
	if ((arg = arg_exists(argc, argv, "--verbose")) != 0) 
	{
		verbose(1);
		printf("Detail text mode enabled\n");
	}				 
	
       	if ((arg = arg_exists(argc, argv, "--help")) != 0) 
	{
		printf("jpecho [--debug][--verbose]\n\n");
		return(0);	// early exit
	}				 
	
	// open driver momentarily to determine how many cards
	{
	  int htemp = vpb_open(1,1);
	  ports_per_card=vpb_get_ports_per_card();
	  num_ch = ports_per_card*vpb_get_num_cards();
	  vpb_close(htemp);
	}
	h = new int[num_ch];
	aport_thread = new pthread_t[num_ch];
	
	// now init each port and start one thread per port
	for(i=0; i<num_ch; i++) {
		h[i] = vpb_open(1+(i/ports_per_card),(i%ports_per_card)+1);
		vpb_sethook_sync(h[i],VPB_ONHOOK);
		vpb_record_set(h[i], &record);
		pthread_create(&aport_thread[i], NULL, port_thread, 
			       (void*)&h[i]);
	}
	printf("%d ports opened!\n", num_ch);

	// do nothing in main loop until return pressed
	while(!kbhit()){
		vpb_sleep(500);
	}
	finito = 1;

	printf("shutting down....\n");
	while(threads_active) {
		vpb_sleep(100);
		printf("threads active: %d\r", threads_active);
	}

	for(i=0; i<num_ch; i++) {
		vpb_close(h[i]);
	}
	delete h;

	return 0;
}

void *port_thread(void *pv)
{
	int     	h = *(int*)pv;
	int     	i, ev;
	char    	file_name[VPB_MAX_STR];
	short		buf[N];	
	VPB_CID_JP      *my_did = new VPB_CID_JP;
	int		state;
	void            *w;
	char 		didfname[VPB_MAX_STR] = {"did00_000.wav"};
	char		itnum[16];
	int		cycle=0;
	ulong		t,buf_i;
	VPB_EVENT	e;
	int		ret;
	
	pthread_mutex_lock(&mutex);
	threads_active++;
	pthread_mutex_unlock(&mutex);
	sprintf(file_name,"linear%02d.wav",h);	// setup audio filename

	didfname[4] = h + 'A';	// Make name unique for each ch.(A..L)

	while(!finito) {
                        printf("Waiting for Fast Ring...\n");
		if(VPB_RING == (ev= ctwaitforevent(h, VPB_RING, 0)))
			ev = ctwaitforevent(h, VPB_RING, 2000);
		if (ev == VPB_RING) 
	       	{
			vpb_sethook_sync(h,VPB_OFFHOOK);
			printf("Reading DID...\n");
		        vpb_record_buf_start(h, VPB_LINEAR);
	        	//vpb_record_buf_sync(h, (char *)buf, sizeof(short)*N);
			t=N;
			buf_i=0;
			while(t>0)
			{
	        		vpb_record_buf_sync(h, (char *)buf+(buf_i*sizeof(short)), sizeof(short)*400);
				t-=400;
				buf_i+=400;
				ret=vpb_get_event_ch_async(h,&e);
				if((ret==VPB_OK) && (e.type==VPB_TONEDETECT) && (e.data==8))
					if(t>(250*8))
						t=(250*8);	// 250ms
			}
		        vpb_record_buf_finish(h);
                        vpb_sethook_sync(h, VPB_ONHOOK);
			if(did_debug ==1)
			{
				sprintf(itnum,"%03d",cycle);
				didfname[6] = itnum[0];	// Make number per iteration.
				didfname[7] = itnum[1];	
				didfname[8] = itnum[2];	
				printf("Saving DID to file [%s].\n",didfname);
				vpb_wave_open_write(&w, didfname, VPB_LINEAR);
				vpb_wave_write(w, (char*)buf, sizeof(short)*N);
				vpb_wave_close_write(w);
				cycle++;
			}
		        //state = vpb_cid_jp_decode(my_did, (short*)buf, N/sizeof(short));
		        state = vpb_cid_jp_decode(my_did, (short*)buf, buf_i);
		}
		if (ev == VPB_TIMEREXP)
		{	state =-1;	}
		if(!finito)
		{
		    printf("Call return status = %d\n\n",state);
		    if((state ==0) && (!finito))
		    {
		        printf("Decoded raw data Presentation Layer ...\n");
		        for(i=0; i<my_did->raw_size  ;i++)
		                printf("%2x,",my_did->raw[i]);
		        printf(" \n");
			printf(" \nJP DID fields\n");
		        printf("==================================================================\n");
		        printf("Origenators Telephone Number   my_did->otn    = %s\n",my_did->otn);
		        printf("OTA Reason for Absence         my_did->otnra  = %s\n",my_did->otnra);
		        printf("OTA Expansion Data             my_did->otnexp = %s\n",my_did->otnexp);
		        printf("DID Called Number Data         my_did->cnd    = %s\n",my_did->cnd);
		        printf("DID Number Expansion Data      my_did->cndexp = %s\n",my_did->cndexp);
			printf("==================================================================\n");
		    }
		    else
			printf("Error returned, No data to display.\n");

		    // Do normal answer even if DID was not passed, else it gets stuck in the forst loop
		    // because the timer expires there before the second slow ring.
		    //if (VPB_RING == (ev= ctwaitforevent(h, VPB_RING, 0))) 
		    do{
		    	ev= ctwaitforevent(h, VPB_RING, 0);
		    }while(ev != VPB_RING);
		    ev = ctwaitforevent(h, VPB_RING, 6000);
		    if (ev == VPB_RING)
		    {
			vpb_sethook_sync(h,VPB_OFFHOOK);
			printf("[%02d] Audio Recording.....\n",h);
			vpb_record_file_sync(h, file_name, VPB_LINEAR);

			printf("[%02d] Audio Playing.....\n",h);
			vpb_play_file_sync(h, file_name);

			vpb_sethook_sync(h, VPB_ONHOOK);
			printf("[%02d] done.....\n",h);
		    }
		}
	}

	pthread_mutex_lock(&mutex);
	threads_active--;
	pthread_mutex_unlock(&mutex);

	return NULL;
}
	
int ctwaitforevent(int h, int event_type, int timeout_ms) {
	char      s[VPB_MAX_STR];
	int       ret, state;
	VPB_EVENT e;   
	void      *timer;

	vpb_timer_open(&timer, h, 0, timeout_ms);
	if (timeout_ms != 0)
		vpb_timer_start(timer);

	state = 1;
	while(state && !finito) {
		ret = vpb_get_event_ch_async(h, &e);

		if (ret == VPB_OK) {
			vpb_translate_event(&e, s);
			printf("%s",s);

			if (e.type == event_type) {
				state = 0;
			}
			if (e.type == VPB_TIMEREXP) {
				state = 0;
			}
			if(e.type == VPB_STATION_OFFHOOK) {
				state = 0;
			}	
		}
		else
			vpb_sleep(100);
	}
	  
	vpb_timer_close(timer);	
	if (state == 0)
		return e.type;
	else
		return -1;
}

int arg_exists(int argc, char *argv[], char *arg) {
  int i;

  for(i=0; i<argc; i++)
    if (strcmp(argv[i],arg) == 0)
      return i;

  return 0;
}

	





