/*---------------------------------------------------------------------------*\

    FILE....: ECHO.CPP
    TYPE....: C++ Console program
    AUTHOR..: David Rowe
    DATE....: 6/12/00

    This program is used for testing Voicetronix.  It works like this:

    1. It waits for ringing on any channel.
    2. It then takes the channel off hook and records for 4 seconds.  You can
       also stop recording by pressing any digit.
    3. It then plays back the recorded speech and hangs up.
    4. Press return to finish, otherwise it waits for the next ring (step 1)

    Compile: make echo
    Run....: ./echo 

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "../src/vpbapi.h"
#include "../src/verbose.h"
#include "kbhit.h"
#include "threads.h"

int ctwaitforevent(int h, int event_type, int timeout_ms);
void *port_thread(void *pv);
int arg_exists(int argc, char *argv[], char *arg);

int             threads_active; // the number of active threads.
pthread_mutex_t mutex;          // mutex to protect shared data
int             finito;         // flag to signal the program is finished.
float           gain;           // option software record gain
int		LV=0;

int main(int argc, char * argv[])
{
	int	        num_ch,*h,i,j,k,ports_per_card;
	VPB_RECORD	record;
	pthread_t       *aport_thread;

	// verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);
	record.term_digits = "1234567890*#";  // any digit terminates record
	record.time_out = 4000;               // record finishes after 4 seconds
	pthread_mutex_init(&mutex,NULL);
	finito = 0;

	// open driver momentarily to determine how many cards
	{
	  int htemp = vpb_open(1,1);
	  ports_per_card=vpb_get_ports_per_card();
	  num_ch = ports_per_card*vpb_get_num_cards();
	  vpb_close(htemp);
	}
	h = new int[num_ch];
	aport_thread = new pthread_t[num_ch];
	
	// now init each port and start one thread per port
	for(i=0; i<num_ch; i++) {
		h[i] = vpb_open(1+(i/ports_per_card),(i%ports_per_card)+1);
		vpb_sethook_sync(h[i],VPB_ONHOOK);
		vpb_record_set(h[i], &record);
		j=arg_exists(argc, argv, "--hookthreshold");
		if((j != 0) && ((j+1) <argc)) 
		{
			k = atoi(argv[j+1]);
			if((k>2) && (k<128))
			{
				vpb_loopvolt_set_threshold(h[i], k);
				LV=1;
			}
			else
				printf("Range error for hook threshold (%d)\n",k);
		}
		j=arg_exists(argc, argv, "--hookwindow");
		if((j != 0) && ((j+1) <argc)) 
		{
			k = atoi(argv[j+1]);
			LV=1;
			vpb_loop_set_onhookwindow(h[i], k);
			vpb_loop_set_offhookwindow(h[i], k);
		}

		pthread_create(&aport_thread[i], NULL, port_thread, 
			       (void*)&h[i]);
	}
	printf("%d ports opened!\n", num_ch);

	// do nothing in main loop until return pressed
	while(!kbhit()){
		vpb_sleep(500);
	}
	finito = 1;

	printf("shutting down....\n");
	while(threads_active) {
		vpb_sleep(100);
		printf("threads active: %d\r", threads_active);
	}

	for(i=0; i<num_ch; i++) {
		vpb_close(h[i]);
	}
	delete h;

	return 0;
}

void *port_thread(void *pv) {
	int       h = *(int*)pv;
	int       ev;
	char      file_name[VPB_MAX_STR];
	short	volts;
	int	ms;

	pthread_mutex_lock(&mutex);
	threads_active++;
	pthread_mutex_unlock(&mutex);
	sprintf(file_name,"linear%02d.wav",h);
	if(1==LV){
		vpb_loop_get_onhookwindow(h, &ms);
		printf("[%02d] onhook window = %d\n",h, ms);
		vpb_loop_get_offhookwindow(h, &ms);
		printf("[%02d] offhook window = %d\n",h, ms);
		vpb_loopvolt_get_threshold(h, &volts);
		printf("[%02d] hook threshold = %d\n",h, (signed char)volts);
	}
	while(!finito) {

		if(VPB_RING == (ev= ctwaitforevent(h, VPB_RING, 0)))
			ev = ctwaitforevent(h, VPB_RING, 6000);
		if ((ev == VPB_RING) || (ev == VPB_STATION_OFFHOOK)) {
			vpb_sethook_sync(h,VPB_OFFHOOK);

			printf("[%02d] recording.....\n",h);
			vpb_record_file_sync(h, file_name, VPB_LINEAR);

			printf("[%02d] playing.....\n",h);
			vpb_play_file_sync(h, file_name);

			vpb_sethook_sync(h, VPB_ONHOOK);
			printf("[%02d] done.....\n",h);
		}
	}

	pthread_mutex_lock(&mutex);
	threads_active--;
	pthread_mutex_unlock(&mutex);

	return NULL;
}
	
int ctwaitforevent(int h, int event_type, int timeout_ms) {
	char      s[VPB_MAX_STR];
	int       ret, state;
	VPB_EVENT e;   
	void      *timer;
	short	  volts;

	vpb_timer_open(&timer, h, 0, timeout_ms);
	if (timeout_ms != 0)
		vpb_timer_start(timer);

	state = 1;
	while(state && !finito) {
		ret = vpb_get_event_ch_async(h, &e);

		if (ret == VPB_OK) {
			vpb_translate_event(&e, s);
			printf("%s",s);

			if (e.type == event_type) {
				state = 0;
			}
			if (e.type == VPB_TIMEREXP) {
				state = 0;
			}
			if(e.type == VPB_STATION_OFFHOOK) {
				state = 0;
			}
			if(1==LV){
			vpb_loopvolt_get(h, &volts);
			printf("[%02d] Hook Level = %d\n",h, (signed char)volts);
			}
		}
		else
			vpb_sleep(100);
		if((h==0) && (1==LV)){
			vpb_loopvolt_get(h, &volts);
			printf("[%02d]                                Hook Level = %d\015",h, (signed char)volts);
		}
	}
	  
	vpb_timer_close(timer);	
	if (state == 0)
		return e.type;
	else
		return -1;
}

int arg_exists(int argc, char *argv[], char *arg) {
  int i;

  for(i=0; i<argc; i++)
    if (strcmp(argv[i],arg) == 0)
      return i;

  return 0;
}

	





