/*---------------------------------------------------------------------------*\

    FILE....: TVPB.CPP
    TYPE....: C++ Program
    AUTHOR..: David Rowe
    DATE....: 23/8/00

    Test program for VPB driver, allows you record and playback short
    messages.  Note this test program is for "Bayonne mode", see 
    serc/vpbapi.cpp for more info on Bayoone mode.

    1. Install the VPB4 and test with 'memtest' (see INSTALL).
    2. Connect a phone line to the port closes to teh top of the VPB4 bracket.
    3. Start this program './tvpb'.
    4. Dial the phone line connected to the VPB4.
    5. Instructions will be printed (along with a lot of diagnostics).
    6. Hit return to exit.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>
#include <string.h>

#include "../../src/vpbapi.h"
#include "../../src/verbose.h"

#define BOARD         1     // board ID for polling loop 
#define POLLING_DELAY 20    // polling loop delay in ms
#define BUF_SZ        32000 // size of sample buffer in bytes 

// states
#define ONHOOK        0
#define IDLE          1
#define RECORD        2
#define PLAY          3

// extra events
#define PLAY_FINISH   100
#define RECORD_FINISH 101

int vpb_set_event_mask(int,unsigned long);
void *event_thread(void *pv);
void *record_thread(void *pv);
void *play_thread(void *pv);
void dispatch_event(VPB_EVENT e);

int       finito;    // used by main to signal event thread to terminate
int       h;         // handle of open port (aka channel, device)
pthread_t record_th, // record thread handle
	  play_th;   // play thread handle 
int       state;     // current state machine state
char     buf[BUF_SZ];// spech sample buffer

// mutex to protect event dispatcher
pthread_mutex_t mutex;

// states in string form
char *state_str[] = {"On Hook", "Idle", "Record", "Play"};

int main() {
	int       iobase[] = {0x310};
	pthread_t thread;

	// initialise
	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);
	vpb_start(1, iobase, "../firmware/vpbmain_isa.out", VPB_VPB4);
	finito = 0;
	state = ONHOOK;
	h = vpb_open(1,1);
	vpb_set_event_mask(h, VPB_MRING | VPB_MDTMF);

	pthread_mutex_init(&mutex, NULL);

	// start event handler thread
	pthread_create(&thread, NULL, event_thread, NULL);
	printf("Press return to finish\n");

	// wait for keypress to exit
	getchar();

	// signal thread to shut down
	finito = 1;
	while(finito) 
		vpb_sleep(POLLING_DELAY);

	pthread_mutex_destroy(&mutex);
	vpb_close(h);
	vpb_stop();
	return 0;
}

// IMPORTANT: vpb_get_event_sync must be called regularly, as driver uses
// this function to service time critical tasks in the driver.  This
// function must be polled regularly, even if no events are required for 
// the program.  Without it, other functions, such as play, record, will
// not work.

void *event_thread(void *pv) {
	VPB_EVENT e;
	char      s[VPB_MAX_STR];

	while(!finito) {
		if (vpb_get_event_async_bayonne(BOARD, &e) == VPB_OK) {
			vpb_translate_event(&e, s);
			printf("%s", s);
			dispatch_event(e);
		}
		vpb_sleep(POLLING_DELAY);
	}

	// signal main that we have finished
	finito = 0;

	return NULL;
}

void dispatch_event(VPB_EVENT e) {
	int  next_state;

	pthread_mutex_lock(&mutex);
        
	// iterate state machine

	printf("dispatch: ev type = %d\n",e.type);

	next_state = state;
	switch (state) {
	case ONHOOK:
		if (e.type == VPB_RING) {
			vpb_sethook_async(h,VPB_OFFHOOK);
			printf("Press 1 - record, 2 - play, 3 - hangup\n");
			next_state = IDLE;
		}
		break;
	case IDLE:
		if (e.type == VPB_DTMF) {

			if (e.data == '1') {
				pthread_create(&record_th, NULL, record_thread,
					       NULL);
				printf("Press 1 - stop record, 2 - hangup\n");
				next_state = RECORD;
			}

			if (e.data == '2') {
				pthread_create(&play_th, NULL, play_thread, 
					       NULL);
				printf("Press 1 - stop play, 2 - hangup\n");
				next_state = PLAY;
			}

			if (e.data == '3') {
				vpb_sethook_async(h,VPB_ONHOOK);
				next_state = ONHOOK;
			}
		}
		break;
	case RECORD:
		if (e.type == VPB_DTMF) {
			if (e.data == '1') {
				vpb_record_terminate(h);
			}

			if (e.data == '2') {
				vpb_record_terminate(h);
				vpb_sethook_async(h,VPB_ONHOOK);
				next_state = ONHOOK;
			}
		}
		if (e.type == RECORD_FINISH) {
			printf("Press 1 - record, 2 - play, 3 - hangup\n");
			next_state = IDLE;
		}
		break;
	case PLAY:
		if (e.type == VPB_DTMF) {
			if (e.data == '1') {
				vpb_play_terminate(h);
			}

			if (e.data == '2') {
				vpb_play_terminate(h);
				vpb_sethook_async(h,VPB_ONHOOK);
				next_state = ONHOOK;
			}
		}
		if (e.type == PLAY_FINISH) {
			printf("Press 1 - record, 2 - play, 3 - hangup\n");
			next_state = IDLE;
		}
		break;
	default:
		assert(0);
		break;

	}

	state = next_state;
	printf("%s\n",state_str[state]);

	pthread_mutex_unlock(&mutex);
}

// thread spawned by event handler to handle recording
void *record_thread(void *pv) {
	VPB_EVENT record_finish = {RECORD_FINISH,0,0,0};

	memset(buf, 0, BUF_SZ);  
	printf("start rec\n");
	vpb_record_buf_start(h, VPB_MULAW);
	vpb_record_buf_sync(h, buf, BUF_SZ);
	vpb_record_buf_finish(h);
	printf("finish rec\n");
	dispatch_event(record_finish);
	return NULL;
}

// thread spawned by event handler to handle playing
void *play_thread(void *pv) {
	VPB_EVENT play_finish = {PLAY_FINISH,0,0,0};

	vpb_play_buf_start(h, VPB_MULAW);
	vpb_play_buf_sync(h, buf, BUF_SZ);
	vpb_play_buf_finish(h);
	dispatch_event(play_finish);
	return NULL;
}


