/*---------------------------------------------------------------------------*\

	FILE....: techophone2.cpp
	TYPE....: C Program
	AUTHOR..: David Rowe
	DATE....: 23/4/01

	Test program to drive DSP echo canceller.  Bridges two
	telephones via the VPB.  The resultant delay should gives the 
	echo canceller a good working out.  

	This program is very similar in function to techophone, but uses the
	high level VPBAPI functions rather than the low level driver 
	functions.

	Intructions:

	1. Connect two lines from the switch to ports 1 & 4.
	2. Run the program.
	3. Using two phones, dial the two ports.
	4. The program will connect the circuit, giving echo free speech if
	   all is well.
	5. Try pressing e<ret> & <d> ret to enable/disable the echo canc.

\*--------------------------------------------------------------------------*/

#include "../../src/mess.h"
#include "../../src/verbose.h"
#include "../../src/vpbapi.h"
#include "../../src/vpbreg.h"
#include "../../src/coff.h"
#include "../../src/comm.h"
#include "../../src/hip.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <pthread.h>
#include <string.h>
// compile with -lgsm if you want this option
//#define GSM
#ifdef GSM
extern "C" {
#include <gsm.h>
}
#endif

#define	BUF_SIZE        160	// size of processing frame (samples) (20ms)
#define NBUF            10      // gives 200ms total one-way delay

// used to pass data to audio threads
typedef struct {
	int   src;
	int   dest;
	float gain;
} PARAM;

// headers for functions internal to driver
int vpb_echo_canc_disable();
int vpb_echo_canc_enable();

// local functions
void *event_thread(void *pv);
void *audio_thread(void *pv);
int kbhit();
int arg_exists(int argc, char *argv[], char *arg);

void vpb_set_codec_reg(int chdev,    // channel handle
		       unsigned short addr,  // 8-bit address of codec register
		       unsigned short data   // 8-bit data to write to register
		      );

int     finish;
int get_adaptc();
extern Comm *vpb_c;
 
int main(int argc, char *argv[])
{
	char		s[VPB_MAX_STR];
	char            c;
	pthread_t       ev_thread, aud_thread1, aud_thread2;
	int             codec, bal, bal2, bal3;
	float           gain;

	int             line1, line2;
	PARAM           param1, param2;

	int             arg;
	short           thresh;
	int             dh;
	float           playg,recg;

	if ((arg = arg_exists(argc, argv, "-swgain")) != 0) {
		param1.gain = param2.gain = atof(argv[arg+1]);
		printf("swgain = %f %f\n",param1.gain,  param2.gain);  
	}
	else {
		param1.gain = param2.gain = 1.0;
	}
	
	// initialise 

	//verbose(1);
	line1 = vpb_open(1,1);
	line2 = vpb_open(1,4);
	vpb_disable_event(line1, VPB_MTONEDETECT);
	vpb_disable_event(line2, VPB_MTONEDETECT);

	if ((arg = arg_exists(argc, argv, "--offhook")) != 0) {
		// take both ports off hook
		vpb_sethook_sync(line1, VPB_OFFHOOK);
		vpb_sethook_sync(line2, VPB_OFFHOOK);		
	}

	if ((arg = arg_exists(argc, argv, "-dial")) != 0) {
		dh = atoi(argv[arg+1]);
		assert((line1==dh) || (line2==dh));
		printf("Dialling %s out handle %d\n",argv[arg+2], dh);
		vpb_sethook_sync(dh, VPB_OFFHOOK);
		vpb_dial_sync(dh, argv[arg+2]);
		
	}

	// option to set codec record gain
	if ((arg = arg_exists(argc, argv, "-rg")) != 0) {
		recg = atof(argv[arg+1]);
		assert((recg >= -12.0) && (recg <= 12.0));
		vpb_record_set_hw_gain(line1, recg);
		vpb_record_set_hw_gain(line2, recg);
	}

	// option to set codec play gain
	if ((arg = arg_exists(argc, argv, "-pg")) != 0) {
		playg = atof(argv[arg+1]);
		assert((playg >= -12.0) && (playg <= 12.0));
		vpb_play_set_hw_gain(line1, playg);
		vpb_play_set_hw_gain(line2, playg);
	}

       	// option to set codec balance reg 1
	if ((arg = arg_exists(argc, argv, "-bal1")) != 0) {
	     	bal = atoi(argv[arg+1]);
		assert((bal >= 0) && (bal <= 255));
		vpb_set_codec_reg(line1, 0x32, bal);
		vpb_set_codec_reg(line2, 0x32, bal);
		printf("setting bal on line 1 only!\n");
	}

	if ((arg = arg_exists(argc, argv, "-bal2")) != 0) {
	     	bal2 = atoi(argv[arg+1]);
		assert((bal2 >= 0) && (bal2 <= 255));
		vpb_set_codec_reg(line1, 0x3a, bal2);
		vpb_set_codec_reg(line2, 0x3a, bal2);
	}

	// option to set codec balance reg 3
	if ((arg = arg_exists(argc, argv, "-bal3")) != 0) {
		bal3 = atoi(argv[arg+1]);
		assert((bal3 >= 0) && (bal3 <= 255));
		vpb_set_codec_reg(line1, 0x42, bal3);
		vpb_set_codec_reg(line2, 0x42, bal3);
	}

	// option to set residual echo suppressor thresh
	if ((arg = arg_exists(argc, argv, "-thresh")) != 0) {
		thresh = atoi(argv[arg+1]);
		assert(thresh >= 0);
		vpb_echo_canc_set_sup_thresh(&thresh);
	}			

	// option to use echo canceller (on by default)
	if (arg_exists(argc, argv, "-nec")) {
		vpb_echo_canc_disable();
	}

	// start main processing loop ---------------------------
		
	printf("E<ret> - enable canceller (default), "
	       "D<ret> - disable, Q<ret> to finish\n"
	       "P codec gain<ret> - set play gain\n"
	       "R codec gain<ret> - set record gain\n"
	       );

	finish = 0;
	pthread_create(&ev_thread, NULL, event_thread, NULL);
	param1.src = line1; param1.dest = line2;
	pthread_create(&aud_thread1, NULL, audio_thread, (void*)&param1);
	param2.src = line2; param2.dest = line1;
	pthread_create(&aud_thread2, NULL, audio_thread, (void*)&param2);
	       
	do {
		fgets(s, VPB_MAX_STR, stdin);
		c = toupper(s[0]);

		switch (c) {
		case 'E':
			// enable echo canceller
			vpb_echo_canc_enable();
			printf("echo canceller ON\n");
			break;
		case 'D':
			// disable echo canceller
			vpb_echo_canc_disable();
			printf("echo canceller OFF\n");
			break;
		case 'P':
			// set play gain
			sscanf(s, "%c %d %f\n",&c, &codec, &gain);
			vpb_play_set_hw_gain(codec, gain);
			printf("codec %d play gain set to %f dB\n",
			       codec, gain);
			break;
		case 'R':
			// set record gain
			sscanf(s, "%c %d %f\n",&c, &codec, &gain);
			vpb_record_set_hw_gain(codec, gain);
			printf("codec %d record gain set to %f dB\n",
			       codec, gain);
			break;
		case 'Q':
			// Quit - finish program
			finish = 1;
			break;
		}

	} while(!finish);

	// clean up and finish 

	vpb_sethook_sync(line1,VPB_ONHOOK);
	vpb_sethook_sync(line2,VPB_ONHOOK);
	
	return 0;
}	

void *event_thread(void *pv) {
	char	  s[VPB_MAX_STR];
        VPB_EVENT e;

	do {
		// check for events from VPB

		while(vpb_get_event_async(&e) == VPB_OK) {
			vpb_translate_event(&e, s);
			mprintf(s);

			// take channels off-hook on ring
			if (e.type == VPB_RING) {
				vpb_sethook_sync(e.handle, VPB_OFFHOOK);
			}
		}

		vpb_sleep(100);
	        #ifdef ADAPTC
		printf("adaptc : %d\n",get_adaptc());
		#endif
	} while(!finish);

	return NULL;
}

// Joins record audio of one channel to play audio of another. Two of
// these threads are required for a full-duplex (two way) conversation. 

void *audio_thread(void *pv) {
	PARAM *param = (PARAM*)pv;
	int   src = param->src;
	int   dest = param->dest;
	float gain = param->gain;
	short buf[NBUF*BUF_SIZE];
	short *pplay,*prec;
	int   i;
	float sam;
        #ifdef GSM
	gsm        handle_in, handle_out;
	gsm_frame  frame;	
	#endif

	vpb_record_buf_start(src, VPB_LINEAR);
	vpb_play_buf_start(dest, VPB_LINEAR);
	prec = buf;
	pplay = &buf[BUF_SIZE];

        #ifdef GSM
	handle_in = gsm_create();
	handle_out = gsm_create();
	#endif

	while(!finish) {
		vpb_record_buf_sync(src, (char*)prec, sizeof(short)*BUF_SIZE);
		for(i=0; i<BUF_SIZE; i++) {
			sam = prec[i]*gain;
			if (sam > 32767.0) sam = 32767.0;
			if (sam < -32767.0) sam = -32767.0;
			prec[i] = (short)sam;
		}
		prec += BUF_SIZE;
		if (prec >= &buf[NBUF*BUF_SIZE])
			prec = buf;
                #ifdef GSM
		gsm_encode(handle_in, pplay, frame);
		gsm_decode(handle_out, frame, pplay);
		#endif
		vpb_play_buf_sync(dest, (char*)pplay, sizeof(short)*BUF_SIZE);
		pplay += BUF_SIZE;
		if (pplay >= &buf[NBUF*BUF_SIZE])
			pplay = buf;
	}

	vpb_record_buf_finish(src);
	vpb_play_buf_finish(dest);

	return NULL;
}

int arg_exists(int argc, char *argv[], char *arg) {
  int i;

  for(i=0; i<argc; i++)
    if (strcmp(argv[i],arg) == 0)
      return i;

  return 0;
}

// adaptc is a counter in the DSP firmware that is incremented for every
// sample that the EC adapts.  The DSP side of the adaptc code is disabled 
// normally but can be switched on (requires firmware recompile).

int get_adaptc()
{
	
		int    b;
		VPBREG *v=vpb_c->vpbreg(0);		
		long   unsigned aadaptc;
		word   adaptc;

		coff_get_address(v->firm, "_adaptc", &aadaptc);
		for(b=0; b<1; b++) {
			vpb_c->hip(b)->ReadDspSram(b, (unsigned short)aadaptc, 1, 
						    (word*)&adaptc);
		}

		return adaptc;
}

