/*---------------------------------------------------------------------------*\

    FILE....: LINEV_API.CPP
    TYPE....: C++ Console program
    AUTHOR..: Peter Wintulich
    DATE....: 10-DEC-2003

    This program is used for testing the V4PCI line voltage detection API calls.

    Compile: gcc linev_api.cpp -lvpb -g -Wall -o linev_api -lm -pthread
    Run....: ./linev_api

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "../../src/vpbapi.h"
#include "../../src/verbose.h"
#include "../kbhit.h"
#include "../threads.h"

// #define CH_PER_CARD 4
#define N           160

int ctwaitforevent(int h, int event_type, int timeout_ms);
void *port_thread(void *pv);
int arg_exists(int argc, char *argv[], char *arg);
int aux(int argc, char * argv[]);

int             threads_active; // the number of active threads.
pthread_mutex_t mutex;          // mutex to protect shared data
int             finito;         // flag to signal the program is finished.
float           gain;           // option software record gain




int main(int argc, char * argv[])
{
	int	i, *h, ret;
	int	htemp,tmp;
	int	num_ch;
	short	volts;
	int	ports_per_card;
        char      s[VPB_MAX_STR];
	VPB_EVENT e;   

	{
	  htemp = vpb_open(1,1);
	  ports_per_card=vpb_get_ports_per_card();
	  num_ch = ports_per_card*vpb_get_num_cards();
	  vpb_close(htemp);
	}
	h = new int[num_ch];

	for(i=0; i<num_ch; i++)
	{	
		h[i] = vpb_open(1+(i/ports_per_card),(i%ports_per_card)+1);
	}

	vpb_sleep(500);
	
	if(vpb_loopvolt_get_lowlimit( h[0],  &volts) == 1)
		printf("Low voltage threshold set at %d\n",volts);
	else
		printf("bus busy or no daughter card\n");
	
	for(i=0; i<num_ch; i++)
	{
		tmp = vpb_loopvolt_get( h[i], &volts);	
		if( tmp ==1)
			printf("Hook volts CH[%d] = %d\n",i,(char)volts);
		else
			printf("bus busy or no daughter card. Return = %d\n",(char)tmp);		
		tmp = vpb_loopvolt_get_threshold( h[i], &volts);
		if( tmp ==1)
			printf("Hook threshold CH[%d] = %d\n",i,volts);
		else
			printf("bus busy or no daughter card. Return = %d\n",(char)tmp);		
	}

	while(!kbhit())
                vpb_sleep(100);


	tmp = vpb_loopvolt_set_threshold(h[0], 20);


	
	for(i=0; i<num_ch; i++)
	{
		tmp = vpb_loopvolt_get( h[i], &volts);	
		if( tmp ==1)
			printf("Hook volts CH[%d] = %d\n",i,(char)volts);
		else
			printf("bus busy or no daughter card. Return = %d\n",(char)tmp);		
		tmp = vpb_loopvolt_get_threshold( h[i], &volts);
		if( tmp ==1)
			printf("Hook threshold CH[%d] = %d\n",i,volts);
		else
			printf("bus busy or no daughter card. Return = %d\n",(char)tmp);		
	}

	while(!kbhit())
	{
		ret = vpb_get_event_async( &e);
		if (ret == VPB_OK) 
		{
			vpb_translate_event(&e, s);
			printf("%s",s);
		}
		else
	                vpb_sleep(100);
	}

	
	for(i=0; i<num_ch; i++)
		vpb_close(h[i]);


	return 0;
}


/*---------------------------------------------------------------------------*\
 *	Subroutines:
 *
 *
 * 
\*---------------------------------------------------------------------------*/
int aux(int argc, char * argv[])
{
	int	        num_ch,*h,i,arg,ports_per_card;
	VPB_RECORD	record;
	pthread_t       *aport_thread;

	if ((arg = arg_exists(argc, argv, "--swgain"))) {
	  gain = (float)atof(argv[arg+1]);
	  printf("Sofware gain = %f dB ", gain);
	  gain = (float)pow(10.0, gain/20.0); // convert from dB to linear
	  printf("(%f linear)\n", gain);
	}
	else
	  gain = 1.0;

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);
	record.term_digits = "";
	record.time_out = 4000; // record finishes after 4 seconds
	pthread_mutex_init(&mutex,NULL);
	finito = 0;

	// open driver momentarily to determine how many cards
	{
	  int htemp = vpb_open(1,1);
	  ports_per_card=vpb_get_ports_per_card();
	  num_ch = ports_per_card*vpb_get_num_cards();
	  vpb_close(htemp);
	}
	h = new int[num_ch];
	aport_thread = new pthread_t[num_ch];
	
	// now init each port and start one thread per port
	for(i=0; i<num_ch; i++) {
		h[i] = vpb_open(1+(i/ports_per_card),(i%ports_per_card)+1);
		vpb_sethook_sync(h[i],VPB_ONHOOK);
		vpb_record_set(h[i], &record);
		pthread_create(&aport_thread[i], NULL, port_thread, 
			       (void*)&h[i]);
	}
	printf("%d ports opened!\n", num_ch);

	while(!kbhit())
		vpb_sleep(100);
	finito = 1;
	printf("shutting down....\n");
	while(threads_active) {
		vpb_sleep(100);
		printf("threads active: %d\r", threads_active);
	}

	for(i=0; i<num_ch; i++) {
		vpb_close(h[i]);
	}
	delete h;

	return 0;
}

void *port_thread(void *pv) {
	int       h = *(int*)pv;
	int       ev, bufs, i, j;
	float     sam;
	void      *wavewr;
	short     bufrd[N];

	pthread_mutex_lock(&mutex);
	threads_active++;
	pthread_mutex_unlock(&mutex);

	while(!finito) {
		if(VPB_RING == (ev= ctwaitforevent(h, VPB_RING, 0)))
			ev = ctwaitforevent(h, VPB_RING, 6000);
		if ((ev == VPB_RING) || (ev == VPB_STATION_OFFHOOK)) {
			vpb_sethook_sync(h,VPB_OFFHOOK);
			printf("[%2d] recording.....\n",h);

			#ifdef NO_SW_GAIN
			vpb_record_file_sync(h, "linear.wav", VPB_LINEAR);
			#else

			vpb_wave_open_write(&wavewr, "linear.wav", VPB_LINEAR);
			vpb_record_buf_start(h, VPB_LINEAR);

			bufs = 4*8000/N;
			for(j=0; j<bufs; j++) {
			  vpb_record_buf_sync(h, (char*)bufrd,sizeof(short)*N);
			  
			  for(i=0; i<N; i++) {
			    sam = (float)bufrd[i];
			    sam *= gain;
			    if (sam > 32767) sam = 32767;
			    if (sam < -32767) sam = -32767;
			    bufrd[i] = (short)sam;
			  }
			  
			  vpb_wave_write(wavewr,(char*)bufrd,sizeof(short)*N);
			}

			vpb_record_buf_finish(h);
			vpb_wave_close_write(wavewr);
			
			#endif

			printf("[%2d] playing.....\n",h);
			vpb_play_file_sync(h, "linear.wav");
			vpb_sethook_sync(h, VPB_ONHOOK);
			printf("[%2d] done.....\n",h);
		}
	}

	pthread_mutex_lock(&mutex);
	threads_active--;
	pthread_mutex_unlock(&mutex);

	return NULL;
}
	
int ctwaitforevent(int h, int event_type, int timeout_ms) {
	char      s[VPB_MAX_STR];
	int       ret, state;
	VPB_EVENT e;   
	void      *timer;

	vpb_timer_open(&timer, h, 0, timeout_ms);
	if (timeout_ms != 0)
		vpb_timer_start(timer);

	state = 1;
	while(state && !finito) {
		ret = vpb_get_event_ch_async(h, &e);

		if (ret == VPB_OK) {
			vpb_translate_event(&e, s);
			printf("%s",s);

			if (e.type == event_type) {
				state = 0;
			}
			if (e.type == VPB_TIMEREXP) {
				state = 0;
			}
			if(e.type == VPB_STATION_OFFHOOK) {
				state = 0;
			}	
		}
		else
			vpb_sleep(100);
	}
	  
	vpb_timer_close(timer);	
	if (state == 0)
		return e.type;
	else
		return -1;
}

int arg_exists(int argc, char *argv[], char *arg) {
  int i;

  for(i=0; i<argc; i++)
    if (strcmp(argv[i],arg) == 0)
      return i;

  return 0;
}

	





