/*--------------------------------------------------------------------------*\

    FILE....: VPBVOX.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 18/6/98

    This file contains the VPB API functions required to control the
    VOX firmware.

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

				INCLUDES
							
\*--------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "comm.h"
#include "config.h"
#include "apifunc.h"
#include "mapdev.h"
#include "vpbvox.h"
#include "vpbapi.h"
#include "vpbhandle.h"
#include "objtrack.h"

/*--------------------------------------------------------------------------*\

			      	DEFINES
							
\*--------------------------------------------------------------------------*/

// Vox mode switches
#define	VOX_SIMPLE		0
#define	VOX_VOLTAGE	      	1

// defaults -----------------------------------------------------------

#define	DEFAULT_MODE		0	// SIMPLE mode as default

// default VOX_SIMPLE params
#define DEFAULT_ONLEVEL		-40.0	// on level	
#define DEFAULT_OFFLEVEL	-50.0	// off level
#define	DEFAULT_RUNON		2000	// runon in ms

// default VOX_VOLTAGE params
#define DEFAULT_ONLEVEL_V	-18.0	// on level	
#define DEFAULT_OFFLEVEL_V	-24.0	// off level
#define	DEFAULT_RUNON_V		30000	// safety runon in ms
#define DEFAULT_TLEVEL_V	-2.0	// transient level	
#define	DEFAULT_TRUNON_V	2000	// transient runon in ms
#define	DEFAULT_WAIT_V		100	// delay in ms

// limits to params

#define	VOXMAX			-2.0	// -2 dB maximum
#define	VOXMIN			-100.0	// -100 dB minimum
#define	VOXRUNMAX		30000	// 30 sec max runon
#define	VOXRUNMIN		50	// 50 ms min runon
#define	VOXMAXLIN		32767	// 0 dB maximum
#define	RUNONMAX		65535   // max runon 65.5 seconds
#define	FS		      	8000	// sampling rate	

typedef struct {
	USHORT	 id;		// object ID
	USHORT	 onlevel;	// signal has to exceed this level
	USHORT	 offlevel;	// countdown starts when signal drops below
	USHORT 	 runon;		// how many frames to count down		
	USHORT   mode;		// vox algorithm mode
	USHORT	 tlevel;
	USHORT	 trunon;
	USHORT   wait;
} VOXCONFIG;

/*---------------------------------------------------------------------------*\

				  GLOBALS
							
\*---------------------------------------------------------------------------*/

static VOXCONFIG *voxconfig;	// ptr to array of VOX structures
				// stores vox config for each channel

/*-------------------------------------------------------------------------*\

                            FUNCTION HEADERS

\*-------------------------------------------------------------------------*/


/*-------------------------------------------------------------------------*\

			       	FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbvox_open()
	AUTHOR...: David Rowe
	DATE.....: 18/6/98

	Initialises the vox module.  This function should be called when the 
	API is initialised.  This function loads up the default VOX parameters
	for each channel.

\*--------------------------------------------------------------------------*/

void vpbvox_open(USHORT numch) {
	int		i;
	word	m[PC_LVOX_UPDATE];
	USHORT	b,ch;
	VPBREG	*v;

	CheckNew(voxconfig = new VOXCONFIG[numch]);	

	// config DSP state variables and mirror in PC

	m[0] = PC_LVOX_UPDATE;
	m[1] = PC_VOX_UPDATE;

	for(i=0; i<numch; i++) {
		maphndletodev(i, &b, &ch);
		v = vpb_c->vpbreg(b);

		voxconfig[i].mode = DEFAULT_MODE;

		if (voxconfig[i].mode == VOX_SIMPLE) {
			voxconfig[i].id = objtrack_handle_to_id(VOXOBJ, i);
			voxconfig[i].onlevel = 
	                (USHORT)(VOXMAXLIN*pow(10.0, DEFAULT_ONLEVEL/20.0));
			voxconfig[i].offlevel = 
			(USHORT)(VOXMAXLIN*pow(10.0, DEFAULT_OFFLEVEL/20.0));
			voxconfig[i].runon = 
			(USHORT)((float)DEFAULT_RUNON/1000)*(FS/v->lsf);
			// for completness
			voxconfig[i].tlevel = 0;
			voxconfig[i].trunon = 0;
			voxconfig[i].wait = 0;
		}
		else {
			voxconfig[i].id = objtrack_handle_to_id(VOXOBJ, i);
			voxconfig[i].onlevel = 
			(USHORT)(VOXMAXLIN*pow(10.0,DEFAULT_ONLEVEL_V/20.0));
			voxconfig[i].offlevel = (USHORT)
			(VOXMAXLIN*pow(10.0, DEFAULT_OFFLEVEL_V/20.0));
			voxconfig[i].runon = (USHORT)
			(((float)DEFAULT_RUNON_V/1000)*(FS/v->lsf));

			voxconfig[i].tlevel = (USHORT)
			(VOXMAXLIN*pow(10.0, DEFAULT_TLEVEL_V/20.0));
			voxconfig[i].trunon = (USHORT)
			(((float)DEFAULT_TRUNON_V/1000)*(FS/v->lsf));
			voxconfig[i].wait = (USHORT)
			(((float)DEFAULT_WAIT_V/1000)*(FS/v->lsf));
		}

		m[2] = voxconfig[i].id;
		m[3] = voxconfig[i].onlevel;
		m[4] = voxconfig[i].offlevel;
		m[5] = voxconfig[i].runon;
		m[6] = voxconfig[i].mode;
		m[7] = voxconfig[i].tlevel;
		m[8] = voxconfig[i].trunon;
		m[9] = voxconfig[i].wait;
		vpb_c->PutMessageVPB(b,m);
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbvox_close()
	AUTHOR...: David Rowe
	DATE.....: 19/6/98

	Closes down vpbvox module.

\*--------------------------------------------------------------------------*/

void vpbvox_close() {
	delete [] voxconfig;	
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_setvox()
	AUTHOR...: David Rowe
	DATE.....: 16/6/98

	Sets the vox parameters for a specified channel.

\---------------------------------------------------------------------------*/

int WINAPI vpb_setvox(int handle, VPB_VOX *vox)
//	int    handle  handle of channel device
//	VOX    *vox    VOX parameters 
{
	USHORT b;      // board number
	USHORT ch;     // channel on board b
	word   m[PC_LVOX_UPDATE];
	VPBREG *v;
	USHORT runon_frames;
	int    frames_sec;
	float  runon_secs;

	// validate

	ValidHandleCheck(handle);
	if (vox->onlevel > VOXMAX) assert(0);
	if (vox->onlevel < VOXMIN) assert(0);
	if (vox->offlevel > VOXMAX) assert(0);
	if (vox->offlevel < VOXMIN) assert(0);
	if (vox->runon > VOXRUNMAX) assert(0);
	if (vox->runon < VOXRUNMIN) assert(0);
	maphndletodev(handle, &b, &ch);
	v = vpb_c->vpbreg(b);
	runon_secs = vox->runon/(float)1000.0;
	frames_sec =  FS/v->lsf;
	runon_frames = (USHORT)(runon_secs*frames_sec);

//	if (runon_frames > RUNONMAX) assert(0);

	// update PC mirror of vox config

	voxconfig[handle].onlevel = (USHORT)
		(VOXMAXLIN*pow(10.0, vox->onlevel/20.0));
	voxconfig[handle].offlevel = (USHORT)
		(VOXMAXLIN*pow(10.0, vox->offlevel/20.0));
	voxconfig[handle].runon = runon_frames;
		
	// update DSPs vox params

	m[0] = PC_LVOX_UPDATE;
	m[1] = PC_VOX_UPDATE;
	m[2] = voxconfig[handle].id;
	m[3] = voxconfig[handle].onlevel;
	m[4] = voxconfig[handle].offlevel;
	m[5] = voxconfig[handle].runon;
	m[6] = VOX_SIMPLE;
	vpb_c->PutMessageVPB(b,m);

	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_getvox()
	AUTHOR...: David Rowe
	DATE.....: 16/6/98

	Gets the vox parameters for a specified channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_getvox(int handle, VPB_VOX *vox)
//	int    handle   handle of channel device
//	VOX    *vox	ptr to returned VOX parameters 
{
	VPBREG *v;
	USHORT b,ch;
	long   runon;

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	v = vpb_c->vpbreg(b);

	vox->onlevel = (float)
		20.0*(float)log10((float)voxconfig[handle].onlevel/VOXMAXLIN);
	vox->offlevel = (float)20.0 *
		(float)log10((float)voxconfig[handle].offlevel/VOXMAXLIN);
	runon = (long)
		((float)voxconfig[handle].runon * 1000.0) / (FS/v->lsf);
	vox->runon = (USHORT)runon;
	
	return(VPB_OK);
}


