/*---------------------------------------------------------------------------*\

    FILE....: VPBDIAL.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 13/2/98

    This file contains the implementation of the vpb_dial API function.
	
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

				 INCLUDES
							
\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

#include "contypes.h"
#include "comm.h"
#include "vpbreg.h"
#include "wobbly.h"
#include "vpbhandle.h"
#include "mapdev.h"
#include "vpbapi.h"
#include "apifunc.h"
#include "vpbdial.h"
#include "objtrack.h"
#include "config.h"
#include "coff.h"
#include "generic.h"
#include "hostdsp.h"

/*---------------------------------------------------------------------------*\

			       DEFINES
							
\*---------------------------------------------------------------------------*/

#define	DTMF_ON		600l	// DTMF pulse in samples
#define	DTMF_OFF	600l	// DTMF space in samples
#define	MAX_DIGITS	100	// max number of digits to dial
#define	DTMF_TONES	16	// number of DTMF tones
#define	DTMF_TONES	16	// number of DTMF tones
#define	PROG_TONES	10	// max number of programmable tones
#define	TOTAL_TONES	(DTMF_TONES+PROG_TONES)

#define	FLASH_SAM	800	// flash time in samples (100ms)
#define	PAUSE		8000l	// pause time in samples (1000ms)
#define	SLEEPMS		20      // sleep time in ms

#define	FS		8000	// sample rate in Hz
#define	FMAX		4000	// maximum frequency tone
#define	LMAX		-1	// 0dB maximum tone level

// tonestate array states

#define	IDLE		0	// tone generator idling
#define	PLAY		1	// tone generator playing
#define	PLAYTONE	2	// tone generator playing single tone

/*---------------------------------------------------------------------------*\

			               GLOBALS
							
\*---------------------------------------------------------------------------*/

// Prototype tone message describing DTMF digit 0, this is modified for other 
// digits

static word dtmf[] = {
	PC_LTONEG, PC_TONEG,
	0,		        // low tone set at run time
	0,     		        // high tone set at run time	
	0,     		        // not used			
	12000,		        // low tone gain (-11dBV into Euro load)
	17000,		        // high tone gain (-9dBV into Euro load)
	0,
	HIGH(DTMF_ON),		// on time			
	LOW(DTMF_ON),
	HIGH(DTMF_ON+DTMF_OFF),	// total time			
	LOW(DTMF_ON+DTMF_OFF),
	0          	        // DSP TONEG object ID, set at run time	
};

// pause message, basically a tone with 0 amplitude

static word pause[] = {
	PC_LTONEG, PC_TONEG,
        0,	        // low tone set at run time
	0,     		// high tone set at run time	
	0,     		// 3rd (unused) tone
	0,		// low tone gain
	0,		// high tone gain
	0,		// 3rd (unsused) gain
	HIGH(PAUSE),	// on time in samples			
	LOW(PAUSE),
	HIGH(PAUSE),	// off time in samples			
	LOW(PAUSE),
	0          	// object ID			
};

// structure defining message for programmable tone and related character

typedef struct {
	word	mess[PC_LTONEG];
	char	c;
} TONE;

// This array below contains a database of all tones that we can play, 
// comprising (by default) dtmf tones and any user programmed tones
// The messages for each tone are stored in this array, and the array
// is indexed by a character, allowing strings of tones to be played
// using the dial function

static TONE tones[TOTAL_TONES];
static int numtones;			// total number of tones defined

// This structure and the array contain info used to init the messages
// for the DTMF tones.  All of the DTMF digits have the same parameters
// except for the tone frequencies which are loaded into the tones[]
// array at init time.

typedef struct {
	USHORT	low;
	USHORT	high;
	char	c;
} DTMF_DIGIT;

DTMF_DIGIT digits[] = {
	{27978, 19072, '1'},
	{27978, 16324, '2'},
	{27978, 13084, '3'},
	{27978,  9314, 'A'},
	{26955, 19072, '4'},
	{26955, 16324, '5'},
	{26955, 13084, '6'},
	{26955, 9314 , 'B'},
	{25700, 19072, '7'},
	{25700, 16324, '8'},
	{25700, 13084, '9'},
	{25700, 9314 , 'C'},
	{24217, 19072, '*'},
	{24217, 16324, '0'},
	{24217, 13084, '#'},
	{24217, 9314 , 'D'},
};

// hook flash message 

static word hook_flash[] = {
	PC_LCODEC_BREAK,
	PC_CODEC_BREAK,
	0,			// channel to flash
	0,			// number of lsf superframes to flash for
};

// the following ptr points to a local copy of the current DTMF
// string being played.  A local copy is kept because the copy
// passed by the calling function may go out of scope before the
// dial string has finished playing

static char	**dialstr;	        // start of dial string
static char	**dialptr;	        // ptr to current char in dialstr
static USHORT	*tonesleft;		// number of tones still to play
static USHORT	*tonestate;		// number of tones still to play
static USHORT   *async;			// flag for each channel that is true
					// if this is an async call
static int      *term;                  // async terminate flags
static USHORT	NumCh;

/*-------------------------------------------------------------------------*\

			     FUNCTION HEADERS

\*-------------------------------------------------------------------------*/

void dial_next_digit(int chdev, char c);
static void check_freq(unsigned short f);
static void check_level(short l);
void configure_for_tone(int chdev);
void finish_async(int chdev);
void clear_config(int chdev);
void dial_async(int chdev, char *dialstring);

/*-------------------------------------------------------------------------*\

			       FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbdial_change_dtmf_length()
	AUTHOR...: David Rowe
	DATE.....: 23/12/02

	Called before vpb_dial_open() to modify the prototype DTMF digit
	length, e.g. via an environment variable.  This will result in
	all digits being of the modified length.

\*--------------------------------------------------------------------------*/

void vpbdial_change_dtmf_length(int ontime_ms, int offtime_ms) {
	int ontime_samples = ontime_ms*(FS/1000);
	int offtime_samples = offtime_ms*(FS/1000);

	//mprintf("ontime_samples = %d\n", ontime_samples);
	dtmf[8] = 0;
	dtmf[9] = (word)ontime_samples;
	dtmf[10] = 0;
	dtmf[11] = (word)(ontime_samples + offtime_samples);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbdial_open()
	AUTHOR...: David Rowe
	DATE.....: 26/2/98

	Initialises the tone generator and dial module.  This function should
	be called when the API is initialised.  This function loads up the 
	default tone generator parameters for the DTMF tones.

\*--------------------------------------------------------------------------*/

void vpbdial_open(USHORT numch) {
	int	i;

	for(i=0; i<DTMF_TONES; i++) {
		memcpy(&tones[i], dtmf, sizeof(dtmf));
		tones[i].mess[2] = digits[i].low;
		tones[i].mess[3] = digits[i].high;
		tones[i].c = digits[i].c;
	}
	numtones = DTMF_TONES;

	dialstr = new char* [numch];	
	dialptr = new char* [numch];	
	tonesleft = new USHORT[numch];
	tonestate = new USHORT[numch];
	async = new USHORT[numch];
	term = new int[numch];

	for(i=0; i<numch; i++) {
		tonestate[i] = IDLE;
		async[i] = 0;
	}

	hook_flash[3] = FLASH_SAM/vpb_c->vpbreg(0)->lsf;
	NumCh = numch;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbdial_close()
	AUTHOR...: David Rowe
	DATE.....: 25/3/98

	Closes down vpbdial module.

\*--------------------------------------------------------------------------*/

void vpbdial_close() {
	delete [] dialstr;	
	delete [] dialptr;	
	delete [] tonesleft;
	delete [] tonestate;
	delete [] async;
	delete [] term;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbdial_validate()
	AUTHOR...: David Rowe
	DATE.....: 30/7/98

	Validates a dial string.

\*--------------------------------------------------------------------------*/

void vpbdial_validate(char *dialstring) {
	int len, i ,j;

	len = strlen(dialstring);
	if (len > MAX_DIGITS)
		throw Wobbly(VPBAPI_DIAL_TOO_MANY_DIGITS);

	if (len == 0)
		return;

	// check all digits valid

	for(j=0; j<len; j++)
		switch(dialstring[j]) {
			case '&':
			break;
			case ',':
			break;
			default:

				// search for digit in table

				char c = toupper(dialstring[j]);
				int num = -1;
				for(i=0; i<numtones; i++) {
					if (tones[i].c == c)
						num = i;
				}
				if (num < 0)
				       throw Wobbly(VPBAPI_DIAL_INVALID_DIGIT);
			break;
		} // switch
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_dial_async()
	AUTHOR...: David Rowe
	DATE.....: 13/2/98

	Sends a DTMF string down a channel device, accepts the following 
	characters by default, plus user defined chars if defined:

	0 1 2 3 4 5 6 7 8 9 0 * # A B C D

	and:

	&	(hook flash)
	,	(pause)

\*--------------------------------------------------------------------------*/

int WINAPI vpb_dial_async(
			  int  chdev,      // handle
			  char *dialstring // string to dial
)
{

	try {
		ValidHandleCheck(chdev);
		vpbdial_dial_async(chdev, dialstring);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_dial_async"));
	}
	
	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_dial_sync()
	AUTHOR...: David Rowe
	DATE.....: 30/7/98

	Sends a DTMF string down a channel device, accepts the following 
	characters by default, plus user defined chars if defined:

	0 1 2 3 4 5 6 7 8 9 0 * # A B C D

	and:

	&	(hook flash)
	,	(pause)

\*--------------------------------------------------------------------------*/

int WINAPI vpb_dial_sync(
			 int  chdev,      // handle
			 char *dialstring // string to dial
)
{

	try {
		ValidHandleCheck(chdev);
		vpbdial_dial_sync(chdev, dialstring);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_dial_sync"));
	}
	
	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbdial_dial_async()
	AUTHOR...: David Rowe
	DATE.....: 30/7/98

	Internal async dial function.

\*--------------------------------------------------------------------------*/

void vpbdial_dial_async(int chdev, char *dialstr)
{
	async[chdev] = 1;
	dial_async(chdev, dialstr);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbdial_dial_sync()
	AUTHOR...: David Rowe
	DATE.....: 30/7/98

	Internal sync dial function.  Actually calls async version, 
	then waits for it to finish.

\*--------------------------------------------------------------------------*/

void vpbdial_dial_sync(int chdev, char *dialstr)
{
	async[chdev] = 0;
	dial_async(chdev, dialstr);

	// now wait for finish
	
	while((tonestate[chdev] != IDLE) && 
	      !term[chdev]) {
		GenericSleep(SLEEPMS);
	}
	
	tonestate[chdev] = IDLE;
	clear_config(chdev);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: dial_async()
	AUTHOR...: David Rowe
	DATE.....: 30/7/98

	Internal dial function.

\*--------------------------------------------------------------------------*/

void dial_async(int chdev, char *dialstring)
{
	int			len;
	VPB_EVENT		e;

	term[chdev] = 0;
	vpbdial_validate(dialstring);
	len = strlen(dialstring);
	if (len == 0) {
		if (async[chdev]) {
			e.type = VPB_DIALEND;
			e.handle = chdev;
			e.data = 0;
			putevt(&e, 0);
		}
		return;
	}

	// copy to local memory so that digit string stays in scope

	CheckNew(dialstr[chdev] = new char[len+1]);
	strcpy(dialstr[chdev], dialstring);
	tonesleft[chdev] = len;

	// Start the ball rolling with the first tone
	
	if (vpbreg_get_model() == VPB_V12PCI) {
//		#ifndef WIN32  
		#ifdef linux
		HostDSPSetupPlay(chdev);
		#endif
	}
	tonestate[chdev] = PLAY;
	dialptr[chdev] = dialstr[chdev];
	dial_next_digit(chdev, *dialptr[chdev]++);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: dial_next_digit()
	AUTHOR...: David Rowe
	DATE.....: 13/2/98

	Plays one tone in a dial string, including hook flashes and pauses.

\*--------------------------------------------------------------------------*/

void dial_next_digit(int chdev, char c)
//  int		chdev		handle of channel device to generate tone on
//  char    c[]			ASCII digit to play
{
	int		i;
	int		num;
	USHORT	b;			// board number
	USHORT	ch;			// channel on board b
	VPBREG	*v;			// ptr to config infor for this board

	maphndletodev(chdev, &b, &ch);
	v = vpb_c->vpbreg(b);

	switch(c) {
		case '&':
			hook_flash[2] = ch;
			vpb_c->PutMessageVPB(b, hook_flash);
		break;
		case ',':
			pause[12] = objtrack_handle_to_id(TONEOBJ, chdev);
			vpb_c->PutMessageVPB(b, pause);
		break;
		default:

			// search for digit in table

			c = toupper(c);
			num = -1;
			for(i=0; i<numtones; i++) {
				if (tones[i].c == c)
					num = i;
			}
			assert((num >= 0) && (num < TOTAL_TONES));
			tones[num].mess[12] = objtrack_handle_to_id(TONEOBJ,
								    chdev);
			vpb_c->PutMessageVPB(b, tones[num].mess);
		break;
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbdial_process_event
	AUTHOR...: David Rowe
	DATE.....: 25/3/98

	This function is called by the MMQ when a DSP_TONEG or DSP_CODEC_BKEN
	messages are sent from a VPB.  Essentially a callback function.

\*--------------------------------------------------------------------------*/

void vpbdial_process_event(word mess[], USHORT board)
//	word	mess[]	message from VPB
//	USHORT	board	VPB board number
{
	USHORT	chdev;

	if (mess[1] == DSP_TONEG)
		chdev = objtrack_id_to_handle(TONEOBJ, mess[2], board);
	else
		chdev = mapdevtohndle(board,mess[2]);
			
	// terminate function may have stopped tone while this
	// event was being generated by DSP so check status

	if (tonestate[chdev] == PLAY) {
	
		// play next tone in string
	
		if (*dialptr[chdev] != 0)
			dial_next_digit(chdev, *dialptr[chdev]++);

		// when terminate message for all tones received
		// post vpb_dial() terminate event to API Q

		tonesleft[chdev]--;
		if (tonesleft[chdev] == 0) {
			finish_async(chdev);
			delete [] dialstr[chdev];
		}

	}

	// check for play single tone mode

	if (tonestate[chdev] == PLAYTONE)
		finish_async(chdev);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: clear_config
	AUTHOR...: David Rowe
	DATE.....: 12/9/98

	Helper function to return the tone generator to a passive state.

\*--------------------------------------------------------------------------*/

void clear_config(int chdev)
{
	USHORT b,ch;

	maphndletodev(chdev, &b, &ch);
	USHORT stobj = objtrack_handle_to_id(TONEOBJ, chdev);
	config_toneg_reset(vpb_c, b, stobj);	
}
	
/*--------------------------------------------------------------------------*\

	FUNCTION.: finish_async
	AUTHOR...: David Rowe
	DATE.....: 12/9/98

	Helper function to perform finishing up activities for async
	dial.

\*--------------------------------------------------------------------------*/

void finish_async(int chdev) {
	VPB_EVENT e;

	clear_config(chdev);
	tonestate[chdev] = IDLE;

	// emit event

	if (async[chdev]) {
		e.type = VPB_DIALEND;
		e.handle = chdev;
		e.data = 0;
		putevt(&e, 0);
	}

}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbdial_check_arb
	AUTHOR...: David Rowe
	DATE.....: 12/9/98

	This function is called regularly from the MMQ to determine if
	an active tone generator is allowed to keep an output channel.
	If another thread has requested it, it is released.

\*--------------------------------------------------------------------------*/

void vpbdial_check_arb()
{
	USHORT	chdev;

	for(chdev=0; chdev<NumCh; chdev++) {

		// check for vpb_dial....

		if (tonestate[chdev] == PLAY) {	
			if (term[chdev]) {
				finish_async(chdev);
				delete [] dialstr[chdev];
			}
		}

		// check for vpb_playtone...

		if (tonestate[chdev] == PLAYTONE) {
		  if (term[chdev]) {
		    finish_async(chdev);
		  }
		}
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_settone()
	AUTHOR...: David Rowe
	DATE.....: 26/2/98

	Enables user to set a user programmed tone, or replace an existing 
	tone.  Performs translation of user parameters to VPB message.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_settone(
		       char ident, // single char that identifies tone
		       VPB_TONE *t // structure containing tone parameters
)
{
	int			i;
	double		two_pi = 2*acos(-1.0);
	long		ton,toff;

	try {
		// determine if a tone with this identifier already exists

		int exists = 0;
		int index = 0;

		ident = toupper(ident);
		for(i=0; i<numtones; i++) {
			if (tones[i].c == ident) {
			    exists = 1;
				index = i;
			}
		}

		// if ident doesn't exist, make new tone

		if (!exists)
			index = numtones++;
		if (numtones >= TOTAL_TONES)
			throw Wobbly(VPBAPI_DIAL_TOO_MANY_TONES);

		// validate tone parameters
		
		check_freq(t->freq1);
		check_freq(t->freq2);
		check_freq(t->freq3);
		check_level(t->level1);
		check_level(t->level2);
		check_level(t->level3);

		// OK, translate params into message

		tones[index].mess[0] = PC_LTONEG;
		tones[index].mess[1] = PC_TONEG;
		tones[index].mess[2] = (USHORT)(2*cos(t->freq1*two_pi/FS)*
						pow(2,14)+0.5);
		tones[index].mess[3] = (USHORT)(2*cos(t->freq2*two_pi/FS)*
						pow(2,14)+0.5);
		tones[index].mess[4] = (USHORT)(2*cos(t->freq3*two_pi/FS)*
						pow(2,14)+0.5);
		tones[index].mess[5] = (USHORT)(pow(2,15)*
						pow(10.0, t->level1/20.0));
		tones[index].mess[6] = (USHORT)(pow(2,15)*
						pow(10.0, t->level2/20.0));
		tones[index].mess[7] = (USHORT)(pow(2,15)*
						pow(10.0, t->level3/20.0));
		ton = t->ton*FS/1000;
		toff = t->toff*FS/1000;
		tones[index].mess[8] = (USHORT)HIGH(ton);
		tones[index].mess[9] = (USHORT)LOW(ton);
		tones[index].mess[10] = (USHORT)HIGH(ton + toff);
		tones[index].mess[11] = (USHORT)LOW(ton + toff);
		tones[index].mess[12] = 0;	// TONEG obj id set at run time
		tones[index].c = ident;		
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_settone"));
	}
	
	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_gettone()
	AUTHOR...: David Rowe
	DATE.....: 26/2/98

	Enables user to retrieve the parameters of a tone.  Performs 
	translation of user parameters from VPB message.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_gettone(
		       char ident, // single char that identifies tone
		       VPB_TONE *t // structure containing tone parameters
)
{
	int		i;
	double		two_pi = 2*acos(-1.0);

	try {
		// determine if a tone with this identifier exists

		int exists = 0;
		int index = 0;

		ident = toupper(ident);
		for(i=0; i<numtones; i++) {
			if (tones[i].c == ident) {
			    exists = 1;
				index = i;
			}
		}

		// if ident doesn't exist, error

		if (!exists)
			throw Wobbly(VPBAPI_DIAL_NO_TONE);

		// OK, translate message to params

		t->freq1 = (USHORT)(acos((float)tones[index].mess[2]/
					 (pow(2,14)*2))*FS/two_pi);
		t->freq2 = (USHORT)(acos((float)tones[index].mess[3]/
					 (pow(2,14)*2))*FS/two_pi);
		t->freq3 = (USHORT)(acos((float)tones[index].mess[4]/
					 (pow(2,14)*2))*FS/two_pi);
		t->level1 = (short)(20*log10((float)tones[index].mess[5]/
					     pow(2,15)));
		t->level2 = (short)(20*log10((float)tones[index].mess[6]/
					     pow(2,15)));
		t->level3 = (short)(20*log10(1E-32+(float)tones[index].mess[7]/
					     pow(2,15)));
		t->ton = ((long)tones[index].mess[8]<<16) + 
			(long)tones[index].mess[9];
		t->ton /= (long)FS/1000;
		t->toff = ((long)tones[index].mess[10]<<16) + 
			(long)tones[index].mess[11];
		t->toff /= (long)FS/1000;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_gettone"));
	}
	
	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: check_freq()
	AUTHOR...: David Rowe
	DATE.....: 26/2/98

	Frequency validation function.

\*--------------------------------------------------------------------------*/

static void check_freq(unsigned short f)
//	unsigned short	f
{
	if (f > FMAX)
		throw Wobbly(VPBAPI_DIAL_INVALID_FREQUENCY);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: check_level()
	AUTHOR...: David Rowe
	DATE.....: 26/2/98

	Level validation function.

\*--------------------------------------------------------------------------*/

static void check_level(short l)
//	short	f	Level in dB with maximum of 0dB
{
	if (l > LMAX)
		throw Wobbly(VPBAPI_DIAL_INVALID_LEVEL);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbdial_playtone()
	AUTHOR...: David Rowe
	DATE.....: 10/9/98

	Plays a user defined tone immediately, internal version.

\*--------------------------------------------------------------------------*/

void vpbdial_playtone(int chdev, VPB_TONE *t)
{
	double		two_pi = 2*acos(-1.0);
	long		ton,toff;
	word		mess[PC_LTONEG];

	ValidHandleCheck(chdev);

	// this assert will fire if you try to start one vpb_playtone_async
	// before another has finished.
	assert(tonestate[chdev] == IDLE); 

	// validate tone parameters
		
	check_freq(t->freq1);
	check_freq(t->freq2);
	check_freq(t->freq3);
	check_level(t->level1);
	check_level(t->level2);
	check_level(t->level3);

	// OK, translate params into message

	mess[0] = PC_LTONEG;
	mess[1] = PC_TONEG;
	mess[2] = (USHORT)(2*cos(t->freq1*two_pi/FS)*pow(2,14)+0.5);
	mess[3] = (USHORT)(2*cos(t->freq2*two_pi/FS)*pow(2,14)+0.5);
	mess[4] = (USHORT)(2*cos(t->freq3*two_pi/FS)*pow(2,14)+0.5);
	mess[5] = (USHORT)(pow(2,15)*pow(10.0, t->level1/20.0));
	mess[6] = (USHORT)(pow(2,15)*pow(10.0, t->level2/20.0));
	mess[7] = (USHORT)(pow(2,15)*pow(10.0, t->level3/20.0));
	ton = t->ton*FS/1000;
	toff = t->toff*FS/1000;
	mess[8] = (USHORT)HIGH(ton);
	mess[9] = (USHORT)LOW(ton);
	mess[10] = (USHORT)HIGH(ton + toff);
	mess[11] = (USHORT)LOW(ton + toff);
	mess[12] = objtrack_handle_to_id(TONEOBJ,chdev);

	// send message to DSP to start playing

	USHORT b,ch;	
	maphndletodev(chdev, &b, &ch);
	if (vpbreg_get_model() == VPB_V12PCI) {
//		#ifndef WIN32
		#ifdef linux
		HostDSPSetupPlay(chdev);
		#endif
	}
	vpb_c->PutMessageVPB(b, mess);


	tonestate[chdev] = PLAYTONE;
	term[chdev] = 0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_playtone_state()
	AUTHOR...: Ben Kramer
	DATE.....: 19/03/03

	Returns state of play tone (1= playing, 0 = not playing)

\*--------------------------------------------------------------------------*/

int WINAPI vpb_playtone_state(int handle)
{

	if(tonestate[handle] == IDLE)
		return(0);
	else
		return(1);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_playtone_async()
	AUTHOR...: David Rowe
	DATE.....: 10/9/98

	Plays a user defined tone immediately, API version.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_playtone_async(int handle, VPB_TONE *vpb_tone)
{

	try {
		async[handle] = 1;
		vpbdial_playtone(handle, vpb_tone);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_playtone_async"));
	}
	
	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_playtone_sync()
	AUTHOR...: David Rowe
	DATE.....: 10/9/98

	Plays a user defined tone immediately, API version.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_playtone_sync(int handle, VPB_TONE *vpb_tone)
{

	try {
		async[handle] = 0;
		vpbdial_playtone(handle, vpb_tone);

		// now wait for finish

		while((tonestate[handle] != IDLE) && !term[handle]) {
			GenericSleep(SLEEPMS);
		}
		clear_config(handle);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_playtone_sync"));
	}
	
	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_tone_terminate()
	AUTHOR...: David Rowe
	DATE.....: 12/3/02

	Terminates playing of tones (programmable or DTMFs).

\*--------------------------------------------------------------------------*/

int WINAPI vpb_tone_terminate(int handle)
{
  term[handle] = 1;
  return VPB_OK;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_set_flash()
	AUTHOR...: David Rowe
	DATE.....: 15/2/04

	Sets the duration of the hook flash in ms.  The hook flash duration
	applies to all channel on all boards.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_set_flash(int flashTime_ms)
{
	int flash_samples;

	flash_samples = flashTime_ms*(FS/1000);
	hook_flash[3] = flash_samples/vpb_c->vpbreg(0)->lsf;
	return VPB_OK;
}

