/*---------------------------------------------------------------------------*\

    FILE....: TONEG.C
    TYPE....: C Module
    AUTHOR..: David Rowe
    DATE....: 5/11/97

    Programmable tone generator functions.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include "toneg.h"
#include "osc.h"
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

/*--------------------------------------------------------------------------*\

			     TYPEDEFS

\*--------------------------------------------------------------------------*/

typedef struct {
    USHORT id;		/* ID of this TONEG object			*/

    OSC	   osc1;	/* params of first oscillator			*/
    OSC	   osc2;	/* params of second oscillator			*/
    OSC	   osc3;	/* params of second oscillator			*/
    long   duration;	/* num samples on				*/
    long   total;	/* total on + off time in samples		*/

    long   t;		/* number samples output so far			*/
    USHORT busy;	/* asserted if tone generator busy		*/

    void   *buf;	/* next tone (one tone buffer)			*/
    int    buf_full;	/* asserted if buffer full			*/
} TG;

/*--------------------------------------------------------------------------*\

			     FUNCTION HEADERS

\*--------------------------------------------------------------------------*/

void get_ready(void *tgv, word mess[]);
static long lmin(long a, long b);
static long lmax(long a, long b);

/*--------------------------------------------------------------------------*\

				FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: toneg_open
	AUTHOR...: David Rowe
	DATE.....: 5/11/97

	Initialisation function for programmable tone generator module.

\*--------------------------------------------------------------------------*/

void toneg_open(void **tgv, USHORT id)
/* void    **tgv;	call progress info structure			*/
/* USHORT  id;		object ID for this tone generator	 	*/
{
   TG      *tg;

   *tgv = (TG*)malloc(sizeof(TG));
   tg = *tgv;
   assert(tg != NULL);

   tg->id = id;
   tg->t = 0;
   tg->busy = 0;

   /* allocate buffer TG */

   tg->buf = (TG*)malloc(sizeof(TG));
   assert(tg->buf != NULL);
   tg->buf_full = 0;

   /* init buffer TG */

   tg = (TG*)tg->buf;
   tg->buf = tg;
   tg->id = id;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: toneg_close
	AUTHOR...: David Rowe
	DATE.....: 5/11/97

	Frees memory associated with programmable tone generator.

\*--------------------------------------------------------------------------*/

void toneg_close(void *tgv)
/* void    *tgv;	call progress info structure	*/
{
   TG      *tg = (TG*)tgv;

   assert(tg->buf != NULL);
   free(tg->buf);
   assert(tg != NULL);
   free(tg);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: toneg_start_tone
	AUTHOR...: David Rowe
	DATE.....: 6/11/97

	Start generating tone described in message.  This function supports
	a one level buffer such that the next tone can be initialised
	while th current one is playing.  This allows seamless continuation
	between tones so that tones of any cadence can be constructed by
	concatenating the required number of component tones.

\*--------------------------------------------------------------------------*/

void toneg_start(void *tgv, word mess[])
/* void    *tgv;	call progress info structure		*/
/* word    *mess[];	message containing tone parameters	*/
{
   TG      *tg = (TG*)tgv;

   assert(tg != NULL);

   /* if tone already playing a tone try to buffer */

   if (tg->busy) {

       /* if buffer full throw a wobbly */

	   assert(tg->buf_full==0);

       /* otherwise fill buffer */

       get_ready(tg->buf, mess);
       tg->buf_full = 1;
       return;
   }

   /* if no tone init primary structure */

   get_ready(tgv, mess);
   tg->busy = 1;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: toneg_reset
	AUTHOR...: David Rowe
	DATE.....: 12/9/98

	Resets a tone generator, used to terminate a tone early.

\*--------------------------------------------------------------------------*/

void toneg_reset(void *tgv, word mess[])
/* void    *tgv;	call progress info structure		*/
/* word    *mess[];	message containing tone parameters	*/
{
   TG      *tg = (TG*)tgv;

   assert(tg != NULL);
   //assert(mess[1] == PC_TONEG_RESET);

   tg->busy = 0;
   tg->buf_full = 0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: get_ready
	AUTHOR...: David Rowe
	DATE.....: 7/11/97

	Initialises the tone generator structure based on the message
	contents.

\*--------------------------------------------------------------------------*/

void get_ready(void *tgv, word mess[])
/* void    *tgv;	call progress info structure		*/
/* word    *mess[];	message containing tone parameters	*/
{
   TG      *tg = (TG*)tgv;

   assert(tg != NULL);

   tg->duration = ((long)mess[8])<<16;
   tg->duration += (long)mess[9];
   assert(tg->duration > 0);

   tg->total = ((long)mess[10])<<16;
   tg->total += (long)mess[11];
   assert(tg->total > 0);

   tg->t = 0;

   osc_init(&tg->osc1, (short)mess[2], (USHORT)mess[5]);
   osc_init(&tg->osc2, (short)mess[3], (USHORT)mess[6]);
   osc_init(&tg->osc3, (short)mess[4], (USHORT)mess[7]);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: toneg_play
	AUTHOR...: David Rowe
	DATE.....: 6/11/97

	Plays a tone, by adding it to the buffer of samples.  If no tone
	currently active, then nothing is added.  By adding to the buffer
	rather than overwriting we can mix tone and other signals if desired.

	When the tone finishes, the function returns -1.

\*--------------------------------------------------------------------------*/

int toneg_play(void *tgv, short buf[], USHORT n)
/* void    *tgv;	call progress info structure		*/
/* short   buf[];	tone added to this buffer		*/
{
   TG      *tg = (TG*)tgv;
   long    tone_sam;	/* no. of tone samples to generate	*/
   long    sam;		/* no. samples to play this frame	*/

   assert(tg != NULL);
   assert(buf != NULL);

   /* do nothing if tone-silence cycle finished */

   if (!tg->busy)
       return 0;

   /*
      Three cases:

      1. This frame is all tone.
      2. This frame is all silence.
      3. This frame is part tone, then part silence.
   */

   tone_sam = tg->duration - tg->t;
   tone_sam = lmin(tone_sam, n);
   tone_sam = lmax(tone_sam, 0);
   sam = lmin(tg->total - tg->t, n);

   osc_oscillate(&tg->osc1, buf, (USHORT)tone_sam);
   osc_oscillate(&tg->osc2, buf, (USHORT)tone_sam);
   osc_oscillate(&tg->osc3, buf, (USHORT)tone_sam);

   tg->t += sam;
   assert(tg->t <= tg->total);

   /* return -1 when tone-silence cycle finished */

   if (tg->total == tg->t) {
       tg->busy = 0;

       return -1;
   }
   else {
     return 1;
   }

}

/* sanity check */

void toneg_sanity(void *tgv)
/* void    *tgv;	call progress info structure		*/
{
   TG      *tg = (TG*)tgv;

   assert(tg->busy == 0);
}

static long lmin(long a, long b) {
  if (a<b)
    return a;
  else
    return b;
}

static long lmax(long a, long b) {
  if (a>b)
    return a;
  else
    return b;
}
