/*--------------------------------------------------------------------------*\

    FILE....: pci.cpp
    TYPE....: C++ module
    AUTHOR..: David Rowe
    DATE....: 23/12/01

    Low level interface to kernel mode driver for V12PCI.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         V12PCI CT Card Software

         Copyright (C) David Rowe 2001 

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

			       DEFINES
							
\*---------------------------------------------------------------------------*/

#define SIZE_WD      0x100      // size of V12PCI card memory in bytes

/*---------------------------------------------------------------------------*\

			       INCLUDES
							
\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

#include "pci.h"
#include "vpb_ioctl.h"
#include "generic.h"
#include "mess.h"

/*---------------------------------------------------------------------------*\

				   FUNCTIONS
							
\*---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_open()
	AUTHOR...: David Rowe
	DATE.....: 23/12/01

	Called first to initialise this module.

\*--------------------------------------------------------------------------*/

int pci_open(void) {
	int ret = open("/dev/vpbhp0",O_RDWR);
	if (ret == -1) {
		fprintf(stderr,
			"\nERROR opening OpenSwitch12 kernel mode "
		        "driver /dev/vpbhp0:\n"
		        "- Is the vpbhp.o kernel mode driver installed?\n"
			"- Does the /dev/vpbhp0 device node exist?\n"
			"- Do you have rw permissions for /dev/vpbhp0?\n"
			"- Is an OpenSwitch12 card installed?\n"
			"- Is 'VPB_MODEL=V12PCI' env variable set?\n"
			"\n");
		exit(1);
	}
	return ret;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_close()
	AUTHOR...: David Rowe
	DATE.....: 23/12/01

	Called last to close down this module.

\*--------------------------------------------------------------------------*/

int pci_close(int fd) {
	return close(fd);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_block_write()
	AUTHOR...: David Rowe
	DATE.....: 23/12/01

	Write a block of shorts to a single address in the card memory.

\*--------------------------------------------------------------------------*/

void pci_block_write
(
 int            fd,
 unsigned short board, 
 unsigned short addr, 
 unsigned short length, 
 unsigned short *buf
)
{
	VPBHP_DATA params;
  
	assert(buf != NULL);
	assert(addr < SIZE_WD);

	// set up ioctlparams
	params.length = length;
	params.vpbhp_addr = addr;
	params.pci_num = board;
	params.data = buf;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_BLOCK_WRITE, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_block_read()
	AUTHOR...: David Rowe
	DATE.....: 23/12/01

	Reads a block of shorts from a single address in the card memory.

\*--------------------------------------------------------------------------*/

void pci_block_read
(
 int            fd,
 unsigned short board, 
 unsigned short addr, 
 unsigned short length, 
 unsigned short *buf
)
{
	VPBHP_DATA params;
  
	assert(buf != NULL);
	assert(addr < SIZE_WD);

	// set up ioctlparams
	params.length = length;
	params.vpbhp_addr = addr;
	params.pci_num = board;
	params.data = buf;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_BLOCK_READ, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_write()
	AUTHOR...: David Rowe
	DATE.....: 23/12/01

	Write a single short to a address in the card memory.

\*--------------------------------------------------------------------------*/

void pci_write
(
 int            fd,
 unsigned short board, 
 unsigned short addr, 
 unsigned short data
)
{
	VPBHP_DATA params;
  
	assert(addr < SIZE_WD);

	// set up ioctlparams
	params.length = 1;
	params.vpbhp_addr = addr;
	params.pci_num = board;
	params.data = &data;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_BLOCK_WRITE, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_read()
	AUTHOR...: David Rowe
	DATE.....: 23/12/01

	Reads a single short from the card memory.

\*--------------------------------------------------------------------------*/

void pci_read
(
 int            fd,
 unsigned short board, 
 unsigned short addr, 
 unsigned short *data
)
{
	VPBHP_DATA params;
  
	assert(addr < SIZE_WD);

	// set up ioctlparams
	params.length = 1;
	params.vpbhp_addr = addr;
	params.pci_num = board;
	params.data = data;
  
        assert(ioctl(fd, VPBHP_IOC_PCI_BLOCK_READ, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_wd_en_l()
	AUTHOR...: David Rowe
	DATE.....: 23/12/01

	Resets the Watch Dog enable line, disabling the Watch Dog timer.

\*--------------------------------------------------------------------------*/

void pci_wd_en_l
(
 int            fd,
 unsigned short board 
)
{
	VPBHP_DATA params;

	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = 0;
	params.pci_num = board;
	params.data = NULL;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_WD_EN_L, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_wd_en_h()
	AUTHOR...: David Rowe
	DATE.....: 23/12/01

	Sets the Watch Dog enable line, enabling the Watch Dog timer.

\*--------------------------------------------------------------------------*/

void pci_wd_en_h
(
 int            fd,
 unsigned short board 
)
{
	VPBHP_DATA params;

	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = 0;
	params.pci_num = board;
	params.data = NULL;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_WD_EN_H, &params) == 0);
	// reset the counters
	assert(ioctl(fd, VPBHP_IOC_PCI_WDR_H, &params) == 0);
	vpb_sleep(1);
	assert(ioctl(fd, VPBHP_IOC_PCI_WDR_L, &params) == 0);

}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_wd_reset()
	AUTHOR...: David Rowe
	DATE.....: 23/12/01

	Toggle the WD reset line, reseting the WD timer.  Must be called at 
	least once every 2-4 seconds if WD timer enabled.

\*--------------------------------------------------------------------------*/

void pci_wd_reset
(
 int            fd,
 unsigned short board 
)
{
	VPBHP_DATA params;
  
	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = 0;
	params.pci_num = board;
	params.data = NULL;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_WDR_H, &params) == 0);
	vpb_sleep(1);
	assert(ioctl(fd, VPBHP_IOC_PCI_WDR_L, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_rx_fifo_how_full()
	AUTHOR...: David Rowe
	DATE.....: 28/1/02

	Returns the number of shorts in the RX software FIFO.

\*--------------------------------------------------------------------------*/

int pci_rx_fifo_how_full
(
 int            fd,
 unsigned short board 
)
{
	VPBHP_DATA params;
	int      howfull;
  
	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = 0;
	params.pci_num = board;
	params.data = (unsigned short*)&howfull;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_RX_FIFO_HOW_FULL, &params) == 0);
	
	return howfull;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_tx_fifo_how_empty()
	AUTHOR...: David Rowe
	DATE.....: 28/1/02

	Returns the number of shorts free in the TX software FIFO.

\*--------------------------------------------------------------------------*/

int pci_tx_fifo_how_empty
(
 int            fd,
 unsigned short board 
)
{
	VPBHP_DATA params;
	int      howempty;

	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = 0;
	params.pci_num = board;
	params.data = (unsigned short*)&howempty;

	assert(ioctl(fd, VPBHP_IOC_PCI_TX_FIFO_HOW_EMPTY, &params) == 0);
	return howempty;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_rx_fifo_block_read()
	AUTHOR...: David Rowe
	DATE.....: 28/1/02

	Reads the RX software FIFO.

\*--------------------------------------------------------------------------*/

void pci_rx_fifo_block_read
(
 int            fd,
 unsigned short board, 
 int            length, 
 unsigned short *buf
)
{
	VPBHP_DATA params;
  
	// set up ioctlparams
	params.length = length;
	params.vpbhp_addr = 0;
	params.pci_num = board;
	params.data = buf;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_RX_FIFO_BLOCK_READ, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_tx_fifo_block_write()
	AUTHOR...: David Rowe
	DATE.....: 28/1/02

	Reads the RX software FIFO.

\*--------------------------------------------------------------------------*/

void pci_tx_fifo_block_write
(
 int            fd,
 unsigned short board, 
 int            length, 
 unsigned short *buf
)
{
	VPBHP_DATA params;
  
	// set up ioctlparams
	params.length = length;
	params.vpbhp_addr = 0;
	params.pci_num = board;
	params.data = buf;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_TX_FIFO_BLOCK_WRITE, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_khook_read()
	AUTHOR...: David Rowe
	DATE.....: 6/8/02

	Reads the kernel mode hook detector running average.

\*--------------------------------------------------------------------------*/

int pci_khook_read(int fd, unsigned short board, int ch) {
	VPBHP_DATA params;
	int      hook;

	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = ch;
	params.pci_num = board;
	params.data = (unsigned short*)&hook;

	assert(ioctl(fd, VPBHP_IOC_PCI_READ_KHOOK, &params) == 0);
	return hook;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_kring_read()
	AUTHOR...: David Rowe
	DATE.....: 28/8/02

	Reads the kernel mode ring detector.  Note that calling this
	function automatically resets the ring detector flag.

	Returns non-zero for a ring-event, 0 otherwise.

\*--------------------------------------------------------------------------*/

int pci_kring_read(int fd, unsigned short board, int ch) {
	VPBHP_DATA params;
	int      ring;

	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = ch;
	params.pci_num = board;
	params.data = (unsigned short*)&ring;

	assert(ioctl(fd, VPBHP_IOC_PCI_READ_KRING, &params) == 0);
	return ring;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_ee_mandata_read()
	AUTHOR...: Peter Wintulich
	DATE.....: 23/12/01

	Reads and translate card data from EEProm on OpenSwitch card,
       	returning data in VPB_MAN_DATA.

\*--------------------------------------------------------------------------*/

void pci_ee_mandata_read(
 	int            fd,
 	unsigned short board, 
 	VPB_MAN_DATA *md
)
{
	USHORT	eebuf[128];	
	VPBHP_DATA params;
	char    hex[17]={"0123456789abcdef"};   // translation table
  
	// assert(NULL != (int)md);

	// set up ioctlparams
	params.length = 6;
	params.vpbhp_addr = 50;
	params.pci_num = board;
	params.data = eebuf;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_BLOCK_EEREAD, &params) == 0);

	        md->date[0]=hex[((eebuf[0]>>12)& 0x0f)];
	        md->date[1]=hex[((eebuf[0]>>8)& 0x0f)];
	        md->date[2]='/';
	        md->date[3]=hex[((eebuf[0]>>4)& 0x0f)];     //month
	        md->date[4]=hex[(eebuf[0] & 0x0f)];
	        md->date[5]='/';
	        md->date[6]=hex[((eebuf[1]>>12)& 0x0f)];    //year
	        md->date[7]=hex[((eebuf[1]>>8)& 0x0f)];
	        md->date[8]=hex[((eebuf[1]>>4)& 0x0f)];
	        md->date[9]=hex[(eebuf[1] & 0x0f)];
	        md->date[10]=0;
                // Decode Rev
                md->revision[0]=hex[((eebuf[2]>>12)& 0x0f)];    //card
                md->revision[1]=hex[((eebuf[2]>>8)& 0x0f)];
                md->revision[2]='.';
                md->revision[3]=hex[((eebuf[2]>>4)& 0x0f)];     //Rev.
                md->revision[4]=hex[(eebuf[2] & 0x0f)];
                md->revision[5]=0;
                // Decode SN:
                md->serial_n[0]=hex[((eebuf[4]>>12)& 0x0f)];    //SN:high
                md->serial_n[1]=hex[((eebuf[4]>>8)& 0x0f)];
                md->serial_n[2]=hex[((eebuf[4]>>4)& 0x0f)];
                md->serial_n[3]=hex[(eebuf[4] & 0x0f)];
                md->serial_n[4]=hex[((eebuf[5]>>12)& 0x0f)];    //SN:low
                md->serial_n[5]=hex[((eebuf[5]>>8)& 0x0f)];
                md->serial_n[6]=hex[((eebuf[5]>>4)& 0x0f)];
                md->serial_n[7]=hex[(eebuf[5] & 0x0f)];
                md->serial_n[8]=0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_kldrop_read()
	AUTHOR...: Ben Kramer
	DATE.....: 30/01/03

	Reads the kernel mode Loop Drop detector.  Note that calling 
	this function automatically resets the detector flag.

	Returns non-zero for an event, 0 otherwise.

\*--------------------------------------------------------------------------*/

int pci_kldrop_read(int fd, unsigned short board, int ch) {
	VPBHP_DATA params;
	int      ring;

	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = ch;
	params.pci_num = board;
	params.data = (unsigned short*)&ring;

//	mprintf("Checking for Loop Drop..  \n");
	assert(ioctl(fd, VPBHP_IOC_PCI_READ_KLDROP, &params) == 0);
	return ring;
}


/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_sbridge_on()
	AUTHOR...: Ben Kramer
	DATE.....: 27/02/03

	Bridge a resource

\*--------------------------------------------------------------------------*/

void pci_sbridge_on
(
 int            fd,
 unsigned short board, 
 unsigned short ch, 
 int		board2,
 int		ch2
)
{
	VPBHP_DATA params;

	int other[2];
	other[0]=board2;
	other[1]=ch2;

	// set up ioctlparams
	params.length = 1;
	params.vpbhp_addr = ch;
	params.pci_num = board;
	params.data = (unsigned short *)&other;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_SBRIDGE_ON, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_sbridge_off()
	AUTHOR...: Ben Kramer
	DATE.....: 27/02/03

	Un-Bridge a resource

\*--------------------------------------------------------------------------*/

void pci_sbridge_off
(
 int            fd,
 unsigned short board, 
 unsigned short ch
)
{
	VPBHP_DATA params;
  
	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = ch;
	params.pci_num = board;
	params.data = NULL;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_SBRIDGE_OFF, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_conf_join()
	AUTHOR...: Ben Kramer
	DATE.....: 26/02/03

	Join a conference

\*--------------------------------------------------------------------------*/

void pci_conf_join
(
 int            fd,
 unsigned short board, 
 unsigned short ch, 
 unsigned short data
)
{
	VPBHP_DATA params;
  

	// set up ioctlparams
	params.length = 1;
	params.vpbhp_addr = ch;
	params.pci_num = board;
	params.data = &data;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_CONF_JOIN, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_conf_leave()
	AUTHOR...: Ben Kramer
	DATE.....: 26/02/03

	Join a conference

\*--------------------------------------------------------------------------*/

void pci_conf_leave
(
 int            fd,
 unsigned short board, 
 unsigned short ch
)
{
	VPBHP_DATA params;
  

	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = ch;
	params.pci_num = board;
	params.data = NULL;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_CONF_LEAVE, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_read_buf_stats()
	AUTHOR...: Ben Kramer
	DATE.....: 22/5/03

	Reads the buffer stats

\*--------------------------------------------------------------------------*/

void pci_read_buf_stats
(
 int            fd,
 unsigned short board, 
 int            length, 
 unsigned short *buf
)
{
	VPBHP_DATA params;
  
	// set up ioctlparams
	params.length = length;
	params.vpbhp_addr = 0;
	params.pci_num = board;
	params.data = buf;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_READ_BUF_STATS, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_loopback_on()
	AUTHOR...: Ben Kramer
	DATE.....: 03/06/2003

	Turns on software tx->rx loop back, disabling receiving from 
	hardware. Usefull when testing echo cancelation

\*--------------------------------------------------------------------------*/

void pci_loopback_on
(
 int            fd,
 unsigned short board 
)
{
	VPBHP_DATA params;
  
	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = 0;
	params.pci_num = board;
	params.data = NULL;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_LOOPBACK_ON, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_loopback_off()
	AUTHOR...: Ben Kramer
	DATE.....: 03/06/2003

	Turns off software tx->rx loop back, enabling receiving from 
	hardware. Usefull when testing echo cancelation

\*--------------------------------------------------------------------------*/

void pci_loopback_off
(
 int            fd,
 unsigned short board 
)
{
	VPBHP_DATA params;
  
	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = 0;
	params.pci_num = board;
	params.data = NULL;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_LOOPBACK_OFF, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_hostecho_on()
	AUTHOR...: Ben Kramer
	DATE.....: 23/06/2003

\*--------------------------------------------------------------------------*/

void pci_hostecho_on
(
 int            fd,
 unsigned short port 
)
{
	VPBHP_DATA params;
  
	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = 0;
	params.pci_num = port;
	params.data = NULL;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_HOSTECHO_ON, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_hostecho_off()
	AUTHOR...: Ben Kramer
	DATE.....: 23/06/2003


\*--------------------------------------------------------------------------*/

void pci_hostecho_off
(
 int            fd,
 unsigned short port 
)
{
	VPBHP_DATA params;
  
	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = 0;
	params.pci_num = port;
	params.data = NULL;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_HOSTECHO_OFF, &params) == 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_hostecho_opt()
	AUTHOR...: Ben Kramer
	DATE.....: 24/06/2003


\*--------------------------------------------------------------------------*/

void pci_hostecho_opt
(
 int            fd,
 unsigned short method 
)
{
	VPBHP_DATA params;
  
	// set up ioctlparams
	params.length = 0;
	params.vpbhp_addr = 0;
	params.pci_num = method;
	params.data = NULL;
  
	assert(ioctl(fd, VPBHP_IOC_PCI_HOSTECHO_OPT, &params) == 0);
}


/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_block_iicread()
	AUTHOR...: Peter Wintulich
	DATE.....: 24/06/2003


\*--------------------------------------------------------------------------*/

int pci_block_iicread(
			   int fd, 
			   unsigned short board, 
			   unsigned short addr, 
			   unsigned short length, 
			   unsigned short *buf)
{
	VPBHP_DATA params;
  
	// set up ioctlparams
	params.length = length;
	params.vpbhp_addr = addr;
	params.pci_num = board;
	params.data = buf;
  
	return ioctl(fd, VPBHP_IOC_PCI_BLOCK_IICREAD, &params);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_block_iicwrite()
	AUTHOR...: Peter Wintulich
	DATE.....: 


\*--------------------------------------------------------------------------*/

int pci_block_iicwrite(
			    int fd,
			    unsigned short board, 
			    unsigned short addr, 
			    unsigned short length, 
			    unsigned short *buf)
{
	VPBHP_DATA params;
  
	// set up ioctlparams
	params.length = length;
	params.vpbhp_addr = addr;
	params.pci_num = board;
	params.data = buf;
  
	return ioctl(fd, VPBHP_IOC_PCI_BLOCK_IICWRITE, &params);
}

/*--------------------------------------------------------------------------*\

         FUNCTION.: pci_fsync_read()
         AUTHOR...: David Rowe
         DATE.....: 28/3/03

	 Reads the kernel mode resync variable, used for debugging.

\*--------------------------------------------------------------------------*/

void pci_fsync_read(int fd, unsigned short board, int *resync, int *testpattern)
{
        VPBHP_DATA  params;
        VPBHP_FSYNC vpbhp_fsync;

	// set up ioctlparams
        params.length = 0;
        params.vpbhp_addr = 0;
        params.pci_num = board;
        params.data = (unsigned short*)&vpbhp_fsync;
        assert(ioctl(fd, VPBHP_IOC_PCI_READ_FSYNC, &params) == 0);
        *resync = vpbhp_fsync.fsync_resync;
        *testpattern = vpbhp_fsync.fsync_testpattern;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: pci_read_rawtdm()
	AUTHOR...: Peter Wintulich
	DATE.....: 20/AUG/03

	Reads a block of shorts from the RAW TDM data collected in a previous
       	ISR cycle. A new sample will be collected after this call is complete. 
	
\*--------------------------------------------------------------------------*/

void pci_read_rawtdm
(
  int            fd,
  unsigned short board,
  unsigned short length,
  unsigned short *buf
)
{
	VPBHP_DATA params;

	assert(buf != NULL);
	// assert(addr < SIZE_WD);
	
	// set up ioctlparams
	params.length = length;
	params.vpbhp_addr = 0;
	params.pci_num = board;
	params.data = buf;
	
	assert(ioctl(fd, VPBHP_IOC_PCI_READ_RAWTDM, &params) == 0);
}
