/*---------------------------------------------------------------------------*\

    FILE....: kring.c
    TYPE....: C Function
    AUTHOR..: David Rowe
    DATE....: 16/8/02

    Kernel based ring detection module for V12PCI, links with hda.c
    We need to do ring det in kernel mode to get accurate timing between
    ring det samples.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2002 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include "kring.h"

/*--------------------------------------------------------------------------*\

			    DEFINES

\*--------------------------------------------------------------------------*/

#define V12PCI_PORTS    12    /* number of ports per V12PCI             */
#define	WINDOW          200   /* no of SPERIODS to sample (200ms)       */
#define THRESH_RING     60    /* no. of H samples in window for RING    */
#define THRESH_SIL      0     /* no. of H samples in window for SIL     */

/* ring detector states */

#define RING_L          0     /* idle state                             */
#define COUNT_H         1     /* counting number of H samples           */
#define COUNT_L         2     /* counting number of L samples           */


/*--------------------------------------------------------------------------*\

			    STATICS

\*--------------------------------------------------------------------------*/

typedef struct {
	unsigned short *base2;                    // V12PCI card memory
	int            ring_state[V12PCI_PORTS];  // ring det state   
	int            count[V12PCI_PORTS];       
	int            counth[V12PCI_PORTS];  
	int            ring_evt[V12PCI_PORTS];    // asserted for ring event
} KRING, *PKRING; 

// maps port to addr, bit pairs for ring det for that port

int ringmap[] = {
  7,0,
  7,2,
  7,4,
  7,6,
  8,0,
  8,2,
  8,4,
  8,6,
  9,0,
  9,2,
  9,4,
  9,6
};

/*--------------------------------------------------------------------------*\

	FUNCTION.: kring_open()
	AUTHOR...: David Rowe
	DATE.....: 1/8/02

	Constructor to init hook detection module for a board.

\*--------------------------------------------------------------------------*/

void *kring_open(unsigned short *base2) {
        PKRING kring;
	int    ch;

	kring = kring_malloc(sizeof(KRING));

	for(ch=0; ch<V12PCI_PORTS; ch++) {
		kring->ring_state[ch] = RING_L;
	        kring->ring_evt[ch] = 0;
	}
	kring->base2 = base2;
	
	return((void*)kring);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: kring_close()
	AUTHOR...: David Rowe
	DATE.....: 17/8/02

	Destructor to close down ring detection module.

\*--------------------------------------------------------------------------*/

void kring_close(void *kring) {
        kring_free(kring);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: kring_sample
	AUTHOR...: David Rowe
	DATE.....: 16/8/02

	Called from ISR every 1ms to sample ring bits and iterate ring det
	state machine.

\*--------------------------------------------------------------------------*/

void kring_sample(void *pv_kring, int btb,unsigned long *sigbuffer)
{
	PKRING  kring = (PKRING)pv_kring;
	int     ring, state, next_state, addr, bit;
	int     ch;
        int     *ring_state = kring->ring_state;
        int     *count = kring->count;
        int     *counth = kring->counth;
        int     *ring_evt = kring->ring_evt;

	for(ch=btb; ch<V12PCI_PORTS; ch++) {

		/* sense ring bit in i-th channel */

		addr = ringmap[ch*2];
		bit = 1 << ringmap[ch*2+1];
		ring = kring_readw(kring->base2 + addr);
		ring &= bit;

		/*
		ring=*sigbuffer && (1<<(ch*2));
		*/

		state = ring_state[ch];

		next_state = state;
		switch(state) {

		case RING_L:
			/* Idle State */
			if (ring) {
				next_state = COUNT_H;
				count[ch] = 0;
				counth[ch] = 0; 
			}
			else
				next_state = RING_L;
			break;

		case COUNT_H:
			/* Count number of H samples in sample window */
			if (count[ch]++ < WINDOW) {	    
				if (ring) counth[ch]++;
			}
			else {
				/* sample window finished, make decn */
				if (counth[ch] >= THRESH_RING) {
					/* generate event */
					ring_evt[ch] = 1;
					next_state = COUNT_L;
					count[ch] = 0;
					counth[ch] = 0;
				}
				else {
					next_state = RING_L;
				}
			}
			break;

		case COUNT_L:
			/* Count number of H samples in sample window */
			if (count[ch]++ < WINDOW) {	    
				if (ring) counth[ch]++;
			}
			else {
				/* sample window finished, make decn */
				if (counth[ch] <= THRESH_SIL) {
					next_state = RING_L;
				}
				else {
					next_state = COUNT_L;
					count[ch] = 0;
					counth[ch] = 0;
				}
			}
			break;

		}

		ring_state[ch] = next_state;

	} /* for(ch=0 ... */
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: kring_read()
	AUTHOR...: David Rowe
	DATE.....: 18/8/02

	Called by driver IOCTL to determine if a ring event has occurred.
	After reading, the ring_evt flag is reset.

\*--------------------------------------------------------------------------*/

int kring_read(void *pv_kring, int ch) {
	PKRING  kring = (PKRING)pv_kring;
        int     *ring_evt = kring->ring_evt;
	int     ret;

	ret =  ring_evt[ch];
	ring_evt[ch] = 0;
	return ret;
}

