/*---------------------------------------------------------------------------*\

    FILE....: hostdsp.cpp
    TYPE....: C+ Functions
    AUTHOR..: David Rowe
    DATE....: 13/5/02

    This modules simulates the "DSP" for host processing cards.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2002 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <string.h>

extern "C" {
#include "alawmulaw.h"
#include "dtmf.h"
#include "toneg.h"
#include "toned.h"
}
#include "mess.h"
#include "hostdsp.h"
#include "pci.h"
#include "messages.h"
#include "generic.h"
#include "dspfifo.h"
#include "hdaport.h"

// Open Switch data

int os_base =0;

// maps port to addr, bit pairs for ring det for that port

int ringmap[] = {
  7,0,
  7,2,
  7,4,
  7,6,
  8,0,
  8,2,
  8,4,
  8,6,
  9,0,
  9,2,
  9,4,
  9,6
};

// maps port to addr, bit pairs for hook det for that port

int hookmap[] = {
  7,1,
  7,3,
  7,5,
  7,7,
  8,1,
  8,3,
  8,5,
  8,7,
  9,1,
  9,3,
  9,5,
  9,7
};

// ring detector states

#define RING_SIL  0
#define RING_RING 1

// General

#define HOSTDSP_SLEEP 5
#define RING_SLEEP    2000
#define V12PCI_PORTS  12
#define MAX_STR       256
#define BANKA         8
#define BANKB         12
#define MAX_BOARDS    12

// V12PCI (HDA) registers

#define NUM_OE        4      // 4 Output Enable registers
#define BITS_OE       8      // bits per OE register
#define OE0           2      // OE  7...0 (MSB..LSB)
#define OE1           3      // OE 15...8
#define OE2           4      // OE 23..16
#define OE3           5      // OE 31..24
#define CONTROL_REG   6      // offset of control register

#define FRAME         32    // number of multiplexed samples/frame
#define NBUF          40    // number of frames/buffer
#define BUF           (FRAME*NBUF)

// station port hook detector (and flash detector)

#define ON_HOOK_THRESH   40
#define OFF_HOOK_THRESH  50
#define OFF_HOOK_THRESH_RINGING  50
#define PORT_DET_THRESH  40

#define HOOK_ONHOOK       0
#define HOOK_OFFHOOK      1
#define HOOK_MAYBE_ONHOOK 2
#define HOOK_MAYBE_FLASH  3

#define FLASH_THRESH      750 // max length of hook flash in ms (flash detector)
#define FLASH_DEBOUNCE    150

#define ALAW_CODES 256

// flash generator (used when dialling from loop ports)

#define FS             8000
#define FLASH_ACTIVE   0
#define FLASH_INACTIVE 1

// Ring generator states
#define RINGGEN_IDLE   0
#define RINGGEN_RING   1
#define RINGGEN_PAUSE  2

#define RINGGEN_ONTIME      1000
#define RINGGEN_OFFTIME     3000
#define RINGGEN_MAX_RINGING 4

// values of ring[] used to signal ring gen state machine
#define RINGGEN_START 1
#define RINGGEN_STOP  2

typedef struct {
	int           state;
	unsigned long time;
	int           one_ring;
} RINGGEN;

typedef struct {
	int           state;
	unsigned long start_time;
	unsigned long duration;
} FLASH;

typedef unsigned char uchar;                                                   
static int      finish;
static int      active_threads;
static int      numbd;
static int      numch;
static VPBREG   *v;
static int      pci;
static int      *hook_state;
static long     *hook_time;
static hdaport  *hdaports;
static int      oe[NUM_OE];
static uchar    bitrev_lut[ALAW_CODES];
static GENERIC_CRITICAL_SECTION mutex;
static void     **dtmf_dec;
static void     **toneg;
static void     **toned;
static FLASH    *flash;

static RINGGEN  *ringgen;
static int      *ring;
static int      total_ringing;

int 		vpb_pconf[MAX_BOARDS][BANKB];


void play();

void ProcessAudio();
void SimulateDSP(void *);
void PlayThread(void *pv);
void RecordThread(void *pv);
void DetectRing();
void DetectLdrop();
void DetectHook();
void DetectPortConfig();
void ProcessCommands();
void OpenPorts();
void ClosePorts();
void oe_init();
void oe_set(int bd, int slot);
void oe_reset(int bd, int slot);
void bitrev(uchar a[], int n);                                     
void prepare_rec_buf(char alawbuf[][NBUF], int card);
void prepare_play_buf(char alawbuf[][NBUF], int card);
void GenerateRing();
void GenerateFlash();
void toned_callback(word mess[]);

/*--------------------------------------------------------------------------*\

	FUNCTION.: HostDSPOpen
	AUTHOR...: David Rowe
	DATE.....: 13/5/02

	Constructor to start host DSP processing.

\*--------------------------------------------------------------------------*/

void HostDSPOpen(int anumbd, VPBREG *avr) {
	int   i;
	VPB_MAN_DATA	md;

	mprintf("Initialising Host DSP ...\n");
	pci = pci_open();
	active_threads = 0;
	finish = 0;
	numbd = anumbd;
	
	// set up bitrev lut
	for(i=0; i<ALAW_CODES; i++) {
		bitrev_lut[i] = i;
	}
	bitrev(bitrev_lut, ALAW_CODES);

	v = avr;
	numch = numbd*v->numch;
	hook_state = new int[numch];
	hook_time = new long[numch];
	dtmf_dec = new void*[numch];
	toneg = new void*[numch];
	toned = new void*[numch];
	ring = new int[numch];
	ringgen = new RINGGEN[numch];
	total_ringing = 0;
	flash = new FLASH[numch];
	for(i=0; i<numch; i++) {
		mprintf("Starting channel %d ...\n",i);
		hook_state[i] = HOOK_ONHOOK;
		hook_time[i]= 0;
		dtmf_dec[i] = dtmf_init();
		toneg_open(&toneg[i], i % V12PCI_PORTS);
		//toned_open(&toned[i], i % V12PCI_PORTS, toned_callback);
		//toneg_open(&toneg[i], i );
		toned_open(&toned[i], i , toned_callback);
		ring[i] = 0;
		ringgen[i].state = RINGGEN_IDLE;
		ringgen[i].time = 0;
		flash[i].state = FLASH_INACTIVE;
	}

	// disable WD timer
	for(i=0; i<numbd; i++) {
		pci_wd_reset(pci, i);
		pci_wd_en_l(pci, i);
	}
	
	// collect Card Info Date, Rev. and SN:
	for(i=0; i<numbd; i++) {
		pci_ee_mandata_read(pci,i, &md);
		strcpy(v[i].mdate, md.date);
		strcpy(v[i].revision, md.revision);
		strcpy(v[i].serial_n, md.serial_n);
	}	
	 
	oe_init();
	OpenPorts();
	DetectPortConfig();

	GenericInitializeCriticalSection(&mutex);
	Generic_beginthread(SimulateDSP, 0, NULL);
	//Generic_beginthread(RingGenThread, 0, NULL);
	Generic_beginthread(PlayThread, 0, NULL);
	Generic_beginthread(RecordThread, 0, NULL);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: HostDSPClose
	AUTHOR...: David Rowe
	DATE.....: 13/5/02

	Constructor to stop host DSP processing.

\*--------------------------------------------------------------------------*/

void HostDSPClose() {
	int i=0;

	finish = 1;
	while((active_threads)&&(i<20)){
		GenericSleep(5);
		i++;
	}
	if (i>19){
		mprintf("Check that all threads are closed\n");
	}
	ClosePorts();
	pci_close(pci);
	delete [] hook_state; 
	delete [] hook_time; 
	delete [] ring; 
	delete [] ringgen;
	delete [] flash; 
	for(i=0; i<numch; i++) {
		dtmf_close(dtmf_dec[i]);
		toneg_close(toneg[i]);
		toned_close(toned[i]);
	}
	delete [] dtmf_dec;
	delete [] toneg;
	delete [] toned;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: OpenPorts
	AUTHOR...: David Rowe
	DATE.....: 16/5/02

	Initialises V12PCI ports.

\*--------------------------------------------------------------------------*/

void OpenPorts() {
	int     i;
	int     bd, ch;
	/* DR 15/8/02, causes some RX resyncs if enabled	*/
	// reset all codecs
	pci_write(pci, 0, CONTROL_REG, 0x80);
        pci_write(pci, 0, CONTROL_REG, 0x00);
	

	hdaports = new hdaport[numch];
	for(i=0; i<numch; i++) {
		ch = i % V12PCI_PORTS;
		bd = i / V12PCI_PORTS;
		hdaports[i].init(pci, bd, ch);
		
		hdaports[i].set_tx_timeslot(ch);
		hdaports[i].set_rx_timeslot(ch+V12PCI_PORTS);
		oe_set(bd, ch+V12PCI_PORTS);
		hdaports[i].set_codec_reg(0x32, 199); // codec balance

		// init DSP (audio) FIFOs

		v[bd].rxdf[ch] = new DspFifo(DSPFIFO_UP, v[bd].szrxdf[ch],
					     v[bd].echo_mode);
		//printf("record side writing [%02d] 0x%x\n", ch, (int)v[bd].rxdf[ch]);
		v[bd].txdf[ch] = new DspFifo(DSPFIFO_DOWN, v[bd].sztxdf[ch],
					      v[bd].echo_mode);
		//printf("play side reading [%02d] 0x%x\n", ch, (int)v[bd].txdf[ch]);

		mprintf("DSP [%02d] Audio FIFOs booted OK\n",i);
	}

}

/*--------------------------------------------------------------------------*\

	FUNCTION.: ClosePorts
	AUTHOR...: David Rowe
	DATE.....: 16/5/02

	Shuts down V12PCI ports.

\*--------------------------------------------------------------------------*/

void ClosePorts() {
	int     i;
	int     bd, ch;

	delete [] hdaports;
	for(i=0; i<numch; i++) {
		ch = i % V12PCI_PORTS;
		bd = i / V12PCI_PORTS;
		delete v[bd].rxdf[ch];
		delete v[bd].txdf[ch]; 
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: SimulateDSP
	AUTHOR...: David Rowe
	DATE.....: 13/5/02

	Simulates the DSP on host processing cards:
	- polls card for signalling events
	- audio signal I/O is handled by different threads

\*--------------------------------------------------------------------------*/

void SimulateDSP(void *d) {

	GenericEnterCriticalSection(&mutex);
	active_threads++;
	GenericLeaveCriticalSection(&mutex);

	while(!finish) {
		DetectRing();
		DetectLdrop();
		DetectHook();
		ProcessCommands();
		GenerateRing();
		GenerateFlash();
		GenericSleep(HOSTDSP_SLEEP);		
	}

	GenericEnterCriticalSection(&mutex);
	active_threads--;
	GenericLeaveCriticalSection(&mutex);

	mprintf("Finishing SimulateDSP\n");
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: RingGenThread
	AUTHOR...: Peter Wintulich
	DATE.....: 23/JUL/04

	Sync. the Ring generation with zero crossing of the H/W Ring Gen.

\*--------------------------------------------------------------------------*/

void RingGenThread(void *d) {

	GenericEnterCriticalSection(&mutex);
	active_threads++;
	GenericLeaveCriticalSection(&mutex);

	while(!finish) {
		GenerateRing();
		GenericSleep(HOSTDSP_SLEEP);		
	}

	GenericEnterCriticalSection(&mutex);
	active_threads--;
	GenericLeaveCriticalSection(&mutex);

	mprintf("Finishing RingGenThread\n");
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: PlayThread
	AUTHOR...: David Rowe
	DATE.....: 11/6/02

	Moves audio samples between the kernel mode driver FIFO and the 
	PC FIFOs.  Play side.

\*--------------------------------------------------------------------------*/

void PlayThread(void *pv)
{
	uchar          alaw;
	int            howempty, i, ch, b[MAX_BOARDS];
	unsigned short buf[FRAME];
	char           alawbuf[MAX_BOARDS][V12PCI_PORTS][NBUF];
		
	assert(numbd < MAX_BOARDS);

	// make sure no nasty frame sync pulses (0x100)
	for(i=0; i<FRAME; i++)
		buf[i] = 0;

	GenericEnterCriticalSection(&mutex);
	active_threads++;
	GenericLeaveCriticalSection(&mutex);

	int fd = pci;
	for(i=0; i<MAX_BOARDS; i++)
		b[i] = 0;

	while(!finish) {
		
		for(i=0; i<numbd; i++) {
			
			howempty = pci_tx_fifo_how_empty(fd, i);
			while(howempty > FRAME) {
				// multiplex
				for(ch=0; ch<V12PCI_PORTS; ch++) {
					alaw = alawbuf[i][ch][b[i]];
					alaw = bitrev_lut[alaw];
					buf[ch+V12PCI_PORTS] = alaw;
				}
				b[i]++;
				if (b[i] == NBUF) {
					b[i] = 0;
					prepare_play_buf(alawbuf[i], i);
				}
				
				pci_tx_fifo_block_write(fd, i, FRAME, buf);
				howempty = pci_tx_fifo_how_empty(fd, i);
			}

		}
		
		GenericSleep(5);
	}
	
	GenericEnterCriticalSection(&mutex);
	active_threads--;
	GenericLeaveCriticalSection(&mutex);

	mprintf("Finishing Play Thread\n");
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: prepare_play_buf
	AUTHOR...: David Rowe
	DATE.....: 8/7/02

	Reads bufs of play samples, and formats them ready for playing.
	Also iterates tone generation.

\*--------------------------------------------------------------------------*/

void prepare_play_buf(char alawbuf[][NBUF], int bd)
{
	int   ch, ret;
	short linbuf[NBUF];
	word  mess[DSP_LTONEG];

	// Read play fifos, add in tone, convert to alaw
	for(ch=0; ch<V12PCI_PORTS; ch++) {
		memset(linbuf, 0, sizeof(linbuf));
		v[bd].txdf[ch]->DSPRead( (word*)linbuf, NBUF);

		ret = toneg_play(toneg[bd*V12PCI_PORTS+ch], linbuf, NBUF);
		if (ret == -1) {
			mess[0] = DSP_LTONEG;
			mess[1] = DSP_TONEG;
			mess[2] = ch;
		        v[bd].upmess->Write(mess, mess[0]);	
		}

		alaw_encode(&alawbuf[ch][0], linbuf, NBUF);
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: RecordThread
	AUTHOR...: David Rowe
	DATE.....: 11/6/02

	Moves audio samples between the kernel mode driver FIFO and the 
	PC FIFOs.  Record side.

\*--------------------------------------------------------------------------*/

void RecordThread(void *d) {
	unsigned short buf[FRAME];
	uchar          alaw;
	int            i,ch,howfull,b[MAX_BOARDS];
	char           alawbuf[MAX_BOARDS][V12PCI_PORTS][NBUF];

	GenericEnterCriticalSection(&mutex);
	active_threads++;
	GenericLeaveCriticalSection(&mutex);

	for(i=0; i<MAX_BOARDS; i++)
		b[i] = 0;

	while(!finish) {

		// iterate through all boards

		for(i=0; i<numbd; i++) {

			// audio being "recorded"

			howfull = pci_rx_fifo_how_full(pci, i);
			while (howfull > 2*FRAME) {
				// frame of multiplexed data ready
				pci_rx_fifo_block_read(pci, i, FRAME, buf);
 
				// do we need to sync?

				if (!(buf[0] & 0x100)) {
					mprintf("[%d] resyncing....\n",i);
					while(!(buf[0] & 0x100)) {
						pci_rx_fifo_block_read(pci, i,1,buf);
					}
					pci_rx_fifo_block_read(pci,i,FRAME-1, &buf[1]);
				} 

				// demultiplex
				for(ch=0; ch<V12PCI_PORTS; ch++) {
					alaw = buf[ch] & 0xff;
					alaw = bitrev_lut[alaw];
					alawbuf[i][ch][b[i]] = (char)alaw;
				}
				b[i]++;
				if (b[i] == NBUF) {
					b[i] = 0;
					prepare_rec_buf(alawbuf[i], i);
				}

				howfull = pci_rx_fifo_how_full(pci, i);
			}

		}
		GenericSleep(5);
	}

	GenericEnterCriticalSection(&mutex);
	active_threads--;
	GenericLeaveCriticalSection(&mutex);

	mprintf("Finishing Record Thread\n");
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: prepare_rec_buf
	AUTHOR...: David Rowe
	DATE.....: 25/6/02

	Transfers buffers of recorded samples to the driver FIFOs.  Also
	runs DTMF decoders on each channel.

\*--------------------------------------------------------------------------*/

void prepare_rec_buf(char alawbuf[][NBUF], int bd)
{
	int   ch, digits_ret;
	short linbuf[NBUF];
	char  key[2];
        word  mess[DSP_LDTMF];

	// convert to linear and send to fifos
	for(ch=0; ch<V12PCI_PORTS; ch++) {
		alaw_decode(linbuf, &alawbuf[ch][0], NBUF);
		v[bd].rxdf[ch]->DSPWrite( (word*)linbuf, NBUF);

		// DTMF processing

		digits_ret = dtmf_decoder(dtmf_dec[bd*V12PCI_PORTS+ch], 
					  key, 2, linbuf, NBUF);
		
		if (digits_ret) {
		    // it should be impossible to detect > 1 digit in NBUF
		    // sample buffer
		    assert(digits_ret == 1);
		    //mprintf("ch = %d bd = %d DTMF.....\n", ch, bd);
                    mess[0] = DSP_LDTMF;
                    mess[1] = DSP_DTMF;
                    mess[2] = ch;
                    mess[3] = key[0];
		    v[bd].upmess->Write(mess, mess[0]);	
		}

		// tone decoder processing
		toned_analyse(toned[bd*V12PCI_PORTS+ch], linbuf, NBUF);
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DetectLdrop
	AUTHOR...: Ben Kramer
	DATE.....: 29/1/03

	Based on DetectRing

\*--------------------------------------------------------------------------*/

void DetectLdrop() {
	int            i, bd, ch, ldrop; 
        word           rmess[DSP_LDROP];

	for(i=0; i<numch; i++) {
		ch = i % V12PCI_PORTS;
		bd = i / V12PCI_PORTS;

		ldrop = pci_kldrop_read(pci, (unsigned short)bd, ch);
		if (ldrop) {
			// generate event
			rmess[0] = DSP_LDROP;
			rmess[1] = DSP_DROP;
			rmess[2] = ch;
			v[bd].upmess->Write(rmess, rmess[0]);	
		}
	}
}
/*--------------------------------------------------------------------------*\

	FUNCTION.: DetectRing
	AUTHOR...: David Rowe
	DATE.....: 13/5/02

	Simple ring detector:
	- looks for rising edge
	- emits ring event
	- waits for RING_SLEEP ms before sensing next ring event

	Note: this might need to be replaced by V4PCI-like ring det in
	future, implemented at kernel level to get exact timing.

\*--------------------------------------------------------------------------*/

void DetectRing() {
	int            i, bd, ch, ring; 
        word           rmess[DSP_LCODEC_RING];

	for(i=0; i<numch; i++) {
		ch = i % V12PCI_PORTS;
		bd = i / V12PCI_PORTS;

		ring = pci_kring_read(pci, (unsigned short)bd, ch);
		if (ring) {
			// generate event
			rmess[0] = DSP_LCODEC_RING;
			rmess[1] = DSP_CODEC_RING;
			rmess[2] = ch;
			v[bd].upmess->Write(rmess, rmess[0]);	
		}
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DetectHook
	AUTHOR...: David Rowe
	DATE.....: 14/5/02

	Detects hook status for station ports.

\*--------------------------------------------------------------------------*/

void DetectHook() {
	int		i, bd, ch, state, next_state; 
	int		running;
        word		mess[DSP_LCODEC_HKOFF];
	unsigned long	duration;
	int		off_hook_thresh;

	for(i=0; i<numch; i++) {
		ch = i % V12PCI_PORTS;
		bd = i / V12PCI_PORTS;

		running = pci_khook_read(pci, (unsigned short)bd, ch);
		// used to "tune" HOOK_THRESH during ring (unittest/ringstat)
		/* 
		if (i == 0)
			mprintf("running = %d\n",running);
		*/

		state = hook_state[i];
		next_state = hook_state[i];
		if ( ringgen[i].state == RINGGEN_IDLE){
			off_hook_thresh=OFF_HOOK_THRESH;
		}
		else {
			off_hook_thresh= OFF_HOOK_THRESH_RINGING;
		}

		switch(state) {
		case HOOK_ONHOOK:
			if (running > off_hook_thresh) {
				next_state = HOOK_OFFHOOK;
				
				// stop ringing
				ring[i] = RINGGEN_STOP;
				hdaports[i].set_ring_off();

				hdaports[i].set_audio_on();

				// generate event
				mess[0] = DSP_LCODEC_HKOFF;
				mess[1] = DSP_CODEC_HKOFF;
				mess[2] = ch;
				v[bd].upmess->Write(mess, mess[0]);	
			}
			break;

		case HOOK_OFFHOOK:
			if (running < ON_HOOK_THRESH) {
				next_state = HOOK_MAYBE_ONHOOK;
				hook_time[i] = GenerictimeGetTime();
			}
			break;

		case HOOK_MAYBE_ONHOOK:

			// check we are still on hook
			if (running < ON_HOOK_THRESH) {

				// if on hook for > max flash duration
				// must be a real on hook
				duration = GenerictimeGetTime() - hook_time[i];
				if (duration > FLASH_THRESH) {
					hdaports[i].set_audio_off();
					next_state = HOOK_ONHOOK;
				        // generate event
					mess[0] = DSP_LCODEC_HKON;
					mess[1] = DSP_CODEC_HKON;
					mess[2] = ch;
					v[bd].upmess->Write(mess, mess[0]);
				}
		        }
			else {
				// maybe a flash
				hook_time[i] = GenerictimeGetTime();
				next_state = HOOK_MAYBE_FLASH;
			}
			break;
		case HOOK_MAYBE_FLASH:
			duration = GenerictimeGetTime() - hook_time[i];
			if (duration > FLASH_DEBOUNCE) {
				// check we are still on hook
				if (running < ON_HOOK_THRESH) {
					hdaports[i].set_audio_off();
					next_state = HOOK_ONHOOK;
					// generate event
					mess[0] = DSP_LCODEC_HKON;
					mess[1] = DSP_CODEC_HKON;
					mess[2] = ch;
					v[bd].upmess->Write(mess, mess[0]);
				}
				else {
					// else must have been a flash
					next_state = HOOK_OFFHOOK;
					// generate event
					mess[0] = DSP_LCODEC_FLASH;
					mess[1] = DSP_CODEC_FLASH;
					mess[2] = ch;
					v[bd].upmess->Write(mess, mess[0]);
				}
			}
			break;
		}


		hook_state[i] = next_state;
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: ProcessCommands
	AUTHOR...: David Rowe
	DATE.....: 17/5/02

	Reads commands sent to "DSP" and processes them.

\*--------------------------------------------------------------------------*/

void ProcessCommands() {
	int            i, offset, read, config, lsf_ms, id; 
        word           mess[MAX_STR];

	for(i=0, offset=0; i<numbd; i++, offset += V12PCI_PORTS) {
		read = v[i].dnmess->Read(&mess[0], 1);	
		if (read == OK) {
			read = v[i].dnmess->Read(&mess[1], mess[0]-1);	
			//printf("mess[0] = %d read = %d\n", mess[0], read);
			switch(mess[1]) {
			case PC_CODEC_ONHK:
				assert(mess[0] == PC_LCODEC_ONHK);
				hdaports[offset + mess[2]].set_loop_onhook();
				break;

			case PC_CODEC_OFFHK:
				assert(mess[0] == PC_LCODEC_OFFHK);

				// only allow loop ports to go offhook

				config = hdaports[offset+mess[2]].get_config();
				if (config == HDAPORT_LOOP) {
				  hdaports[offset + mess[2]].set_loop_offhook();
				}
				break;

			case PC_STATION_OFF:
				assert(mess[0] == PC_LSTATION_OFF);
				hdaports[offset + mess[2]].set_audio_off();
				break;

			case PC_STATION_ON:
				assert(mess[0] == PC_LSTATION_ON);
				hdaports[offset + mess[2]].set_audio_on();
				break;

			case PC_RING_ON:
				// cadenced ring gen start
				assert(mess[0] == PC_LRING_ON);
				ringgen[offset + mess[2]].one_ring = mess[3];
				ring[offset + mess[2]] = RINGGEN_START;
				break;

			case PC_RING_OFF:
				// cadenced ring gen stop
				assert(mess[0] == PC_LRING_OFF);
				ring[offset + mess[2]] = RINGGEN_STOP;
				break;

			case PC_URING_OFF:
				assert(mess[0] == PC_LURING_OFF);
				hdaports[offset + mess[2]].set_ring_off();
				break;

			case PC_URING_ON:
				assert(mess[0] == PC_LURING_ON);
				hdaports[offset + mess[2]].set_ring_on();
//				mprintf("ring on %d\n", offset + mess[2]);
				break;

			case PC_SPI_BRIDGE:
				assert(mess[0] == PC_LSPI_BRIDGE);
				hdaports[offset + mess[2]].
					set_rx_timeslot(mess[3]);
				hdaports[offset + mess[3]].
					set_rx_timeslot(mess[2]);
				break;

			case PC_SPI_BRIDGE_OFF:
				assert(mess[0] == PC_LSPI_BRIDGE_OFF);
				hdaports[offset + mess[2]].
					set_rx_timeslot(mess[2]+V12PCI_PORTS);
				hdaports[offset + mess[3]].
					set_rx_timeslot(mess[3]+V12PCI_PORTS);
				break;

			case PC_SPI_SBRIDGE:
				assert(mess[0] == PC_LSPI_SBRIDGE);
				hdaports[offset + mess[2]].
					sbridge_on((int)mess[3],(int)mess[4]);
				break;

			case PC_SPI_SBRIDGE_OFF:
				assert(mess[0] == PC_LSPI_SBRIDGE_OFF);
				hdaports[offset + mess[2]].sbridge_off();
				break;

			case PC_CONF_JOIN:
				assert(mess[0] == PC_LCONF_JOIN);
				hdaports[offset + mess[2]].conf_join(mess[3]);
				break;

			case PC_CONF_LEAVE:
				assert(mess[0] == PC_LCONF_LEAVE);
				hdaports[offset + mess[2]].conf_leave();
				break;

			case PC_CODEC_UPGAIN:
				assert(mess[0] == PC_LCODEC_UPGAIN);
				hdaports[offset + mess[2]].set_tx_gain(
						  (unsigned char)mess[3]);
				break;

			case PC_CODEC_DNGAIN:
				assert(mess[0] == PC_LCODEC_DNGAIN);
				hdaports[offset + mess[2]].set_rx_gain(
						  (unsigned char)mess[3]);
			break;

			case PC_TONEG:
				assert(mess[0] == PC_LTONEG);
				toneg_start(toneg[offset + mess[12]], mess);
			break;

			case PC_TONEG_RESET:
				toneg_reset(toneg[offset + mess[2]], mess);
				break;

			case PC_CODEC_BREAK:
				// timed loop break - generate flash
				lsf_ms = v->lsf*1000/FS;
				flash[offset + mess[2]].duration = lsf_ms*mess[3];
				flash[offset + mess[2]].start_time = GenerictimeGetTime();
				hdaports[offset + mess[2]].set_loop_onhook();
				flash[offset + mess[2]].state = FLASH_ACTIVE;
				mprintf("lsf_ms = %d, duration = %ld\n", lsf_ms, 
					flash[offset + mess[2]].duration);
				break;

			case PC_LOOPBACK_ON:
				assert(mess[0] == PC_LLOOPBACK_ON);
				pci_loopback_on(pci, i);	
				break;

			case PC_LOOPBACK_OFF:
				assert(mess[0] == PC_LLOOPBACK_OFF);
				pci_loopback_off(pci, i);	
				break;

			case PC_HOSTECHO_ON:
				assert(mess[0] == PC_LHOSTECHO_ON);
				pci_hostecho_on(pci,offset+ mess[2]);	
				break;

			case PC_HOSTECHO_OFF:
				assert(mess[0] == PC_LHOSTECHO_OFF);
				pci_hostecho_off(pci,offset+ mess[2]);	
				break;

			case PC_HOSTECHO_OPT:
				assert(mess[0] == PC_LHOSTECHO_OFF);
				pci_hostecho_opt(pci, mess[2]);	
				break;

			case PC_WD_ENABLE:
				assert(mess[0] == PC_LWD_ENABLE);
				if (mess[2]) {
					pci_wd_en_h(pci, i);
				} else {
					pci_wd_en_l(pci, i);
				}
					
				break;

			case PC_WD_RESET:
				assert(mess[0] == PC_LWD_RESET);
				pci_wd_reset(pci, i);	
				break;

			case PC_TONED_ST:  	
				// call progress detector messages 
				id = offset + mess[2];
				//mprintf("Adding tone to toned[%d] mess[%d]\n",id,mess[2]);
				toned_add_tone(toned[id], mess);
				break;

			case PC_TONED_ST_DEL:  	
				// call progress detector messages 
				id = offset + mess[2];
				//mprintf("Adding tone to toned[%d] mess[%d]\n",id,mess[2]);
				toned_del_tone(toned[id], mess);
				break;

			case PC_CODEC_GEN:  	
				// general purpose word sent to codec config reg
				id = offset + mess[2];
				//printf("set codec reg ch %d  reg 0x%x val 0x%x\n", id, mess[3], mess[4]);
				hdaports[id].set_codec_reg(mess[3], mess[4]);
				break;

	                case PC_TONED_DEBUG: 
	                        id = offset + mess[2];
	                        assert(mess[0] == PC_LTONED_DEBUG);
	                        toned_start_logging(toned[id], mess);
	                break;

			default:
				mprintf("unknown message %d\n", mess[1]);
				//assert(0);
				break;
			}
		}
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: HostDSPSetupPlay
	AUTHOR...: David Rowe
	DATE.....: 11/6/02

	Sets up the codec time slots and OE generator for playing to a port.
	Called just before you want to play.

\*--------------------------------------------------------------------------*/

void HostDSPSetupPlay(int h)
{
	int bd, ch;

	ch = h % V12PCI_PORTS;
	bd = h / V12PCI_PORTS;
	//oe_set(bd, ch+V12PCI_PORTS);

	hdaports[h].set_rx_timeslot(ch+V12PCI_PORTS);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: oe_init
	AUTHOR...: David Rowe
	DATE.....: 17/5/02

	Initialises the Output Enable generator.  This generator controls
        which slots the Host writes to on the TDM bus.

\*--------------------------------------------------------------------------*/

void oe_init() {
	int i,j;

	for(j=0; j<numbd; j++) {
		// turn off OE generator on slots 0..31
		for(i=0; i<NUM_OE; i++) {
			oe[i] = 0;
			pci_write(pci, j, OE0+i, 0x00);
		}
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: oe_set
	AUTHOR...: David Rowe
	DATE.....: 17/5/02

	Enables a slot on the OE generator.  The PC will now be outputting
	data to this slot.

\*--------------------------------------------------------------------------*/

void oe_set(int bd, int slot) {
	int reg  = slot/BITS_OE;
	int bit  = slot - reg*BITS_OE;
	int mask = 1 << bit;

	oe[reg] |= mask;
	pci_write(pci, bd, OE0+reg, oe[reg]);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: oe_reset
	AUTHOR...: David Rowe
	DATE.....: 17/5/02

	Disables a slot on the OE generator. 

\*--------------------------------------------------------------------------*/

void oe_reset(int bd, int slot) {
	int reg  = slot/BITS_OE;
	int bit  = slot - reg*BITS_OE;
	int mask = 1 << bit;

	oe[reg] ^= mask;
	pci_write(pci, bd, OE0+reg, oe[reg]);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: bitrev
	AUTHOR...: David Rowe
	DATE.....: 5/6/02

	Bit reversal of samples - hardware bug work around.

\*--------------------------------------------------------------------------*/

void bitrev(uchar a[], int n) {
  int i,j;
  uchar b,c;
 
  for(i=0; i<n; i++) {
    b = a[i];
    c = 0;
    for(j=0; j<8; j++) {
      c |= ((b>>j)&0x1) << (7-j);
    }
    a[i] = c;
  }
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: GenerateRing
	AUTHOR...: David Rowe
	DATE.....: 18/8/02

	Generate cadenced ring signal, called periodically to iterate
	state machine. Prevents more than 4 ports ringing at once, to
	prevent overload of power supply.

\*--------------------------------------------------------------------------*/

void GenerateRing() {
	unsigned long time;
	int bd, ch, i, state, next_state;

	time =  GenerictimeGetTime();	

	for(i=0; i<numch; i++) {
		ch = i % V12PCI_PORTS;
		bd = i / V12PCI_PORTS;
	
		state = ringgen[i].state;
		next_state = state;

		switch(state) {
		case RINGGEN_IDLE:
			if ((ring[i] == RINGGEN_START) && 
			    (total_ringing < RINGGEN_MAX_RINGING)) {
				hdaports[i].set_ring_on();
				ringgen[i].time = time;
				next_state = RINGGEN_RING;
				ring[i] = 0;
				total_ringing++;
			}
			break;
		case RINGGEN_RING:
			if ((time > (ringgen[i].time + RINGGEN_ONTIME))
			    || (ring[i] == RINGGEN_STOP)) {
				hdaports[i].set_ring_off();
				ringgen[i].time = time;
				total_ringing--;

				// ring gen has been told to stop, or one ring flag set
				if ((ring[i] == RINGGEN_STOP) || ringgen[i].one_ring ) {
					next_state = RINGGEN_IDLE;
					ring[i] = 0;
				}
				else
					next_state = RINGGEN_PAUSE;
			}
			break;
		case RINGGEN_PAUSE:
			if (ring[i] == RINGGEN_STOP) {
				next_state = RINGGEN_IDLE;
				ring[i] = 0;
			}
		
			if (time > (ringgen[i].time + RINGGEN_OFFTIME)) {
				hdaports[i].set_ring_on();
				ringgen[i].time = time;
				next_state = RINGGEN_RING;
				total_ringing++;
			}
			break;
		}

		ringgen[i].state = next_state;
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DetectPortConfig
	AUTHOR...: David Rowe
	DATE.....: 22/8/02

	Automatically determines V12PCI configuration of loop/station ports.
	Taking a station port "offhook" will assert the hook detector.

\*--------------------------------------------------------------------------*/

void DetectPortConfig() {
	int            i, bd, offset;
	int            running;
	int		val;
	extern int	vpb_pconf[MAX_BOARDS][BANKB];

	for(bd=0,offset=0; bd<numbd; bd++,offset+=V12PCI_PORTS)
       	{
                // Test all ports and set Station or Loop

               for(i=0; i<BANKB; i++)
               {
                    val=hdaports[offset+i].get_codec_reg(0x6);
                    mprintf("%d\n",val);
                    if(val == 0xa0)    // codec is present
                    {
                                                     
			hdaports[offset+i].set_loop_offhook();
			GenericSleep(50);
			running = pci_khook_read(pci, (unsigned short)bd, i);
			hdaports[offset+i].set_loop_onhook();
			if (running > PORT_DET_THRESH)
		       	{
			  hdaports[offset+i].set_config(HDAPORT_STATION);
	                  mprintf("bd[%d] Port[%d] configured as STATION (running=%d)\n",bd,i,running);			
			  vpb_pconf[bd][i]=HDAPORT_STATION;
			}
			else
		       	{
			  hdaports[offset+i].set_config(HDAPORT_LOOP);
	                  mprintf("bd[%d] Port[%d] configured as LOOP (running=%d)\n", bd,i,running);	
			  vpb_pconf[bd][i]=HDAPORT_LOOP;
			}
		    }
		    else
		    {
		       hdaports[offset+i].set_config(HDAPORT_ABSENT);
		       mprintf("bd[%d] Port[%d] Codec absent\n", bd,i);
		       vpb_pconf[bd][i]=HDAPORT_ABSENT;
		    }
		}

	}
}


/*--------------------------------------------------------------------------*\

	FUNCTION.: GenerateFlash
	AUTHOR...: David Rowe
	DATE.....: 27/8/02

	Generate hook flash while dialing.

\*--------------------------------------------------------------------------*/

void GenerateFlash() {
	unsigned long time, start_time, run_time, duration;
	int           bd, ch, i, state;
	word          mess[DSP_LCODEC_BKEN];

	time =  GenerictimeGetTime();	

	for(i=0; i<numch; i++) {
		ch = i % V12PCI_PORTS;
		bd = i / V12PCI_PORTS;
	
		state = flash[i].state;
		start_time = flash[i].start_time;
		duration = flash[i].duration;
		
		if (state == FLASH_ACTIVE) {
			run_time = time - start_time;
			mprintf("run_time = %ld duration = %ld", run_time, duration);
			if (run_time > duration) {
				hdaports[i].set_loop_offhook();
				state = FLASH_INACTIVE;
				mess[0] = DSP_LCODEC_BKEN;
				mess[1] = DSP_CODEC_BKEN;
				mess[2] = ch;
				v[bd].upmess->Write(mess, mess[0]);			       
			}
		}

		flash[i].state = state;
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: toned_callback
	AUTHOR...: David Rowe
	DATE.....: 14/1/03

	This function is called by the tone detector when a tone is detected
	or a tone detector log message is generated.

\*--------------------------------------------------------------------------*/

void toned_callback(word mess[]) {

	// convert from absolute channel number to bd, ch

	mprintf("GOT TONE FROM[%d]\n",mess[2]);
	int ch = mess[2] % V12PCI_PORTS;
	int bd = mess[2] / V12PCI_PORTS;

	mess[2] = ch;
	v[bd].upmess->Write(mess, mess[0]);		
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: read_iic()
	AUTHOR...: Peter Wintulich
	DATE.....: 18/8/04

        Read I2C port.
\*--------------------------------------------------------------------------*/
int Read_Iic(unsigned short card, unsigned short addr, unsigned short length, unsigned short *buf)
{
	int ret;

	ret = pci_block_iicread(pci, card, addr, length, buf);
	//mprintf("%d = read_iic(0x%04x, ...)\n",ret,addr);
	return (ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: write_iic()
	AUTHOR...: Peter Wintulich
	DATE.....: 18/8/04

	Write I2C port.

\*--------------------------------------------------------------------------*/
int Write_Iic( unsigned short card, unsigned short addr, unsigned short length, unsigned short *buf)
{
	int ret;

	ret = pci_block_iicwrite(pci, card, addr, length, buf);
	//mprintf("%d = write_iic(0x%04x, ...)\n",ret,addr);
	return (ret);
}



