/*--------------------------------------------------------------------------*\

    FILE....: hdaport.cpp
    TYPE....: C++ module
    AUTHOR..: David Rowe
    DATE....: 14/1/02

    Interface to HDA card ports.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         V12PCI CT Card Software

         Copyright (C) David Rowe 2002 

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

			       DEFINES
							
\*---------------------------------------------------------------------------*/

#define HDA_PORTS   12    // number of ports/card
#define CONTROL_REG 6     // offset of control register

/*---------------------------------------------------------------------------*\

			          INCLUDES
							
\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "hdaport.h"
#include "pci.h"
#include "mess.h"

/*---------------------------------------------------------------------------*\

				   FUNCTIONS
							
\*---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: hdaport()
	AUTHOR...: David Rowe
	DATE.....: 14/1/02

	Called first to initialise all ports on the current card.

\*--------------------------------------------------------------------------*/

void hdaport::init(
		 int afd,   // handle of open hda kernel mode driver
		 int acard, // card number
		 int aport  // port number
)
{
	assert(afd != -1);
	assert(acard >= 0);
	assert((aport >= 0) && (aport < HDA_PORTS));

	fd = afd; card = acard; port = aport;

	// init codec

	send_control_word(0x02a0);
	                        // CONTROL REG			
				// codec in normal mode		
				// delayed data timing		
				// even bit inversion A-law	
				// power amp disabled		
				// MCLK = 2.048 MHz		

	send_control_word(0x12e0);
		                // LATCH DIRECTION REG		
				// L0,1,2 outputs	        
				// - L0 ring
	                        // - L1 hook_sw
	                        // - L2 audio_sw

	// init output latch
	latch_bits = 0;
	send_control_word(0x0a00 + latch_bits);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: sbridge_on(int card2, int port2)
	AUTHOR...: Ben Kramer
	DATE.....: 27/02/03

	Bridge a resource

\*--------------------------------------------------------------------------*/
void hdaport::sbridge_on(int card2, int port2) {
	pci_sbridge_on(this->fd, this->card, this->port, card2, port2);
}
/*--------------------------------------------------------------------------*\

	FUNCTION.: sbridge_off()
	AUTHOR...: Ben Kramer
	DATE.....: 27/02/03

	Un-Bridge resource

\*--------------------------------------------------------------------------*/
void hdaport::sbridge_off() {
	pci_sbridge_off(fd, card, port);
}


/*--------------------------------------------------------------------------*\

	FUNCTION.: conf_join(int resource)
	AUTHOR...: Ben Kramer
	DATE.....: 26/02/03

	Joins a conference

\*--------------------------------------------------------------------------*/
void hdaport::conf_join(int resource) {
	pci_conf_join (fd,card, port, (unsigned short)resource);
}
/*--------------------------------------------------------------------------*\

	FUNCTION.: conf_leave()
	AUTHOR...: Ben Kramer
	DATE.....: 26/02/03

	Leaves a conference

\*--------------------------------------------------------------------------*/
void hdaport::conf_leave() {
	pci_conf_leave (fd,card, port);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: send_control_word()
	AUTHOR...: David Rowe
	DATE.....: 14/1/02

	Sends a 16 bit control word to the TS5070 codec.

\*--------------------------------------------------------------------------*/

void hdaport::send_control_word(unsigned short cword) {
	int      i;
	unsigned char val;
	int      data_bit;

	for(i=0; i<16; i++) {
		data_bit = (cword >> (15-i)) & 0x1;
		val = port + (1<<4) + (data_bit<<5) + (1<<6);
		pci_write(fd, card, CONTROL_REG, val);
		val = port + (1<<4) + (data_bit<<5);
		pci_write(fd, card, CONTROL_REG, val);
	}
	
        pci_write(fd, card, CONTROL_REG, 0x0);
}

/*--------------------------------------------------------------------------*\

         FUNCTION.: read_control_byte()
         AUTHOR...: Peter Wintulich
         DATE.....: 09/1/03

         Sends upper byte of 16 bit control word to the TS5070 codec,
         Recives data byte from codec.

\*--------------------------------------------------------------------------*/

int hdaport::read_control_byte(unsigned short cword)
{
        int      i, byte =0;
        unsigned char val;
        int      data_bit;
        short unsigned read_data_bit;

        for(i=0; i<8; i++)      // write Codec port address to CI
        {
                data_bit = (cword >> (15-i)) & 0x1;
                val = port + (1<<4) + (data_bit<<5) + (1<<6);
                pci_write(fd, card, CONTROL_REG, val);  // CCLK=H & data
                val = port + (1<<4) + (data_bit<<5);
                pci_write(fd, card, CONTROL_REG, val);  // CCLK=L
        }
        byte=0;
        for(i=0; i<8; i++)      // Read codec data through CO
	{
	        val = port + (1<<4) + (1<<6);
	        pci_write(fd, card, CONTROL_REG, val);  // clock rise
	        //  may need to add a delay here?
	        val = port + (1<<4);
	        // for(d=0;d<100000;d++);  //vpb_sleep(1);
	        pci_read(fd,card, CONTROL_REG, &read_data_bit); // read bit
	        pci_write(fd, card, CONTROL_REG, val);  // clock fall
	        mprintf("%x,",read_data_bit);
	        byte=(byte<<1) + (int)(read_data_bit & 0x01);   // assemble byte
	}
	pci_write(fd, card, CONTROL_REG, 0x0);
	mprintf("  %d=read_control_byte(%x)\n",byte,cword);
	return byte;
}

/*--------------------------------------------------------------------------*\

         FUNCTION.: read_control_bits()
         AUTHOR...: Peter Wintulich
         DATE.....: 23/7/04

         Read Global status word from card. Used for ring gen. sync.

\*--------------------------------------------------------------------------*/
int hdaport::read_status_word(unsigned short flagmask)
{
        unsigned short read_data_word;

	pci_read(fd,card, CONTROL_REG, &read_data_word); // read word
	mprintf(" 0x%04x=read_status_word()\n",read_data_word);
	return (read_data_word & flagmask ?1:0);
}

/*--------------------------------------------------------------------------*\

         FUNCTION.: read_iic()
         AUTHOR...: Peter Wintulich
         DATE.....: 18/8/04

         Read I2C port.

\*--------------------------------------------------------------------------*/
int hdaport::read_iic(	unsigned short addr,
			unsigned short length,
			unsigned short *buf)
{
        int ret;

	ret = pci_block_iicread(fd, card, addr, length, buf); 
	//mprintf("read_iic(0x%04x, ...)\n",addr);
	return (ret);
}

/*--------------------------------------------------------------------------*\

         FUNCTION.: write_iic()
         AUTHOR...: Peter Wintulich
         DATE.....: 18/8/04

         Write I2C port.

\*--------------------------------------------------------------------------*/
int hdaport::write_iic( unsigned short addr,
			unsigned short length,
			unsigned short *buf)
{
        int ret;

	ret = pci_block_iicwrite(fd, card, addr, length, buf);
	//mprintf("write_iic(0x%04x, ...)\n",addr);
	return (ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTIONS: set and get functions
	AUTHOR...: David Rowe
	DATE.....: 24/1/02

\*--------------------------------------------------------------------------*/

// loop start port - hook sw and audio sw
void hdaport::set_loop_offhook() {
	latch_bits |= 0x60;
	send_control_word(0x0a00+latch_bits);
}

void hdaport::set_loop_onhook() {
	latch_bits &= ~0x60;
	send_control_word(0x0a00+latch_bits);
}

// station port - audio sw only
void hdaport::set_audio_on() {
	latch_bits |= 0x20;
	send_control_word(0x0a00+latch_bits);
}

void hdaport::set_audio_off() {
	latch_bits &= ~0x20;
	send_control_word(0x0a00+latch_bits);
}

void hdaport::set_ring_on() {
	latch_bits |= 0x80;
	send_control_word(0x0a00+latch_bits);
}

void hdaport::set_ring_off() {
	latch_bits &= ~0x80;
	send_control_word(0x0a00+latch_bits);
}
	
void hdaport::set_tx_timeslot(int atimeslot) {
	tx_timeslot = atimeslot;
	send_control_word(0x5280 + tx_timeslot);
}

void hdaport::set_rx_timeslot(int atimeslot) {
	rx_timeslot = atimeslot;
	send_control_word(0x4a80 + rx_timeslot);
}

void hdaport::set_tx_gain(unsigned char gain) {
	send_control_word(0x2a00 + gain);
}

void hdaport::set_rx_gain(unsigned char gain) {
	send_control_word(0x2200 + gain);
}

void hdaport::set_codec_reg(unsigned char reg, unsigned char value) {
	unsigned short high = reg;
	send_control_word( (high<<8) + value);
}

int  hdaport::get_codec_reg(unsigned char reg) {
        unsigned short high = reg; // | 0x4;        // ensure read bit set?
        return read_control_byte( (high<<8) );
}

void hdaport::set_config(int loop_or_station) {
	config = loop_or_station;
}

int hdaport::get_config() {
	return config;
}

