/*--------------------------------------------------------------------------*\

    FILE....: hda.c
    TYPE....: Linux device driver
    AUTHOR..: David Rowe
    DATE....: 22/12/01

    Linux kernel mode device driver for V12PCI (OpenSwitch12) and V6PCI cards.
    This is the main file of the kernel mode driver vpbhp.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         V12PCI CT Card Software

         Copyright (C) David Rowe 2001 david@voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

			       DEFINES
							
\*---------------------------------------------------------------------------*/

#define	NAME	     "vpbhp"
//#define SIZE_WD      0x0100      // size of V12PCI card memory in bytes
#define SIZE_WD      0xffff      // size of V12PCI card memory in bytes, ofr iic
#define MAX_V12PCI    4         // max number of V12PCI cards
#define SIZE_LCR     128        // size of 9052 local config reg space in bytes
#define SIZE_BUF     8096       // max size of V12PCI card block read/write
#define SIZE_FIFO    512        // size of FIFOs on prototype
#define SIZE_SW_FIFO 25600      // 100ms worth of samples
#define SIZE_HW_FIFO 8192	// Maximum amount of data available from HW FIFO

#define EEPROM_SIZE  64		// Eeprom size
#define EEPROM_CS    25		// control bits
#define EEPROM_CLK   24
#define EEPROM_RDBIT 27
#define EEPROM_WRBIT 26

#define IIC_SDA_CTL     6       // USER2 Pin function select
#define IIC_SDA_DIR     7       // USER2 Pin input/output select
#define IIC_SDA_IO      8       // USER2 Pin data in/out
#define IIC_SCL_CTL     9       // USER3 Pin function select
#define IIC_SCL_DIR     10      // USER3 Pin input/output select
#define IIC_SCL_IO      11      // USER3 Pin data in/out

#define PARTY_MAX	10	// Maximum number of parties
#define PARTY_MAXMEM	10	// Maximum number of parties members

#define TX_FIFO_FULL	0x02	// transmit-to-board buffer full
#define TX_FIFO_EMPTY	0x04	// transmit-to-board buffer empty
#define RX_FIFO_FULL	0x08	// transmit-to-board buffer full
#define RX_FIFO_EMPTY	0x10	// transmit-to-board buffer empty

#ifndef MODULE
#define MODULE
#define __KERNEL__
#endif

// Diagnostic and debug tests
// #define TDM_DATA_RAW
//#define TDM_DATA
#define TF_SYNC
// #define TDM_DI

// Different fifo data collection
#define LONG_WORD_RX

/*---------------------------------------------------------------------------*\

			       INCLUDES
							
\*---------------------------------------------------------------------------*/

// Module version boilerplate
#include <linux/autoconf.h>
#if defined( CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#define MODVERSIONS
#endif

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/ioport.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <asm/segment.h>
#include <asm/uaccess.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <asm/system.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <asm/uaccess.h>
/* for rdtsc() */
#include <asm/msr.h>

#include <linux/version.h>
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
#include <linux/interrupt.h>
#endif



#include "vpb_ioctl.h"
//#include "iic.h"              // s/w IIC emulation
#include "iic.c"
#include "fifoc.h"
#include "khook.h"
#include "kring.h"
#include "kldrop.h"
#include "echo.h"
#include "version.h"

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,13)
MODULE_LICENSE("GPL");
#endif

#ifdef VPB_ASTERISK_TIMING
#include "/usr/src/zaptel/zaptel.h"

typedef struct vpb_zt_vpb {
	struct zt_span span;
	struct zt_chan chan;
} vpb_zt_vpb;

static struct vpb_zt_vpb vpb_zt;

static int vpb_zt_init(void);
static int vpb_zt_exit(void);
#endif

 
/* macros to save/restore FPU, this allows floats to be used in ISR */
/* ref: http://www.aero.polimi.it/~rtai/documentation/articles/paolo-floatingpoint.html */

#define save_cr0_and_clts(x) __asm__ __volatile__ ("movl %%cr0,%0; clts" :"=r" (x))
#define restore_cr0(x)       __asm__ __volatile__ ("movl %0,%%cr0": :"r" (x));
#define save_fpenv(x)        __asm__ __volatile__ ("fnsave %0" : "=m" (x))
#define restore_fpenv(x)     __asm__ __volatile__ ("frstor %0" : "=m" (x));

/* code to measure CPU clock cycles, perhaps move to separate module? */

typedef struct {
	u32 lo;
	u32 hi;
} timestamp_t;

void timestamp(timestamp_t *pstamp)
{
	rdtsc(pstamp->lo, pstamp->hi);
}

/* acc = sub - acc */
void timestamp_sub(timestamp_t *sub, timestamp_t *acc)
{
	unsigned long lo = acc->lo - sub->lo;
	unsigned long hi = acc->hi - sub->hi;
	if (lo < 0) {
		lo += 0x10000000L;
		hi--;
	}
	acc->hi = hi;
	acc->lo = lo;
}


/*---------------------------------------------------------------------------*\

				GLOBALS
							
\*---------------------------------------------------------------------------*/

// ensure dynamic major number picked
static unsigned int major = 0;

// Interupt Service Routine Defn
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
irqreturn_t isr(int irq, void *dev_id, struct pt_regs *regs);
#else
void isr(int irq, void *dev_id, struct pt_regs *regs);
#endif
/*
int ldropevt=0;
int new_ldropevt=0;
*/

static int vpbhp_ioctl(struct inode *, struct file *, unsigned int, 
		     unsigned long);

// Bens debug stuff
/*
static int	max_sample_count=0;
static int	min_sample_count=1000;
static int	sample_count_delta=0;
static ulong	sample_count=0;
static int	last_sample_count=0;
*/
static ulong	int_count=0;
static ulong	frame_count=0;
static ulong	incomplete_frame_count=0;
static ulong	junked_samples_count=0;
static ulong	buffer_empty_count=0;

static short	h_tmpbuf[MAX_V12PCI][32];		// history buffer, prev. sample
static short	tmpbuf[MAX_V12PCI][2*SIZE_FIFO];	// holding b. for data from fifo
static short	tmpbufidx[MAX_V12PCI];			// how full is tmpbuf

void 	read_hw_fifo(int bd);

void 	read_hw_fifo_13(int bd);			
static short	holdbuf[MAX_V12PCI][SIZE_HW_FIFO];	// holds aligned data ready to be placed on the rx buff
static short	holdbuf_end[MAX_V12PCI];		// Index to where to put the next data
static short	holdbuf_lastfs[MAX_V12PCI];		// Count to where to the last frame sync is (back from end)
static ulong	hwfifobuf_count_max=0;			// Maximum read from HW FIFO
static ulong	hwfifobuf_count_min=SIZE_HW_FIFO;	// Minimum read from HW FIFO
static ulong	holdbuf_max=0;				// Maximum held in hold buffer
static ulong	holdbuf_min=SIZE_HW_FIFO;		// Minimum held in hold buffer
static ulong	hold_hw_match_under=0;			// Number of times the hold and hw buffers under matched
static ulong	hold_hw_match_over=0;			// Number of times the hold and hw buffers over matched
static ulong	hold_hw_match_not=0;			// Number of times the hold and hw buffers didnt matched

// impulse testing variables
/*
static short	imp_idx=0;
static int	imp_loc_delta[3]={0,0,0};
static int	imp_loc[3]={0,0,0};
static int	imp_loc_last[3]={0,0,0};
static short	imp_sem=1;
*/

// alaw <-> linear stuff
#define ALAW_CODES 256
short linear2alaw(int sample);
int alaw2linear( short alawbyte);
void bitrev(unsigned char a[], int n) ;
static unsigned char    bitrev_lut[ALAW_CODES];
static short linear2alaw_lut[0x10000];
static int alaw2linear_lut[0x100];
#define	LUTOS	32768	

// eeprom access routines
void cntrlbit(int PCInn, int bit, int val);
void wrbit(int PCInn, int bit);
int rdbit(int PCInn);
unsigned int eeread(int PCIn, int addr);
int card_detail(int);
int atoi(char *);

// echo routines and variables
void test_echo(void);
#define N      160
#define AMP    8000
#define	ERL_SIM		0.1	// default 1, to emulate a hybrid turn on loopback
				// and set to 0.1
#define	ECHO_BLOCK	8	// number of samples to collect for ec
#define	MAX_PORTS	12	// maximum number of ports per card
/*
static float ec_y[ECHO_TAPS+N];
static float ec_s[N];
static float ec_e[N];
static void  *ec;
*/
static float	echoy[MAX_V12PCI][MAX_PORTS][ECHO_TAPS+N];
static float	echos[MAX_V12PCI][MAX_PORTS][N];
static float	echoe[MAX_V12PCI][MAX_PORTS][N];
/*
void	*echoy;
void	*echos;
void	*echoe;
*/
//static void *ec_states;
static void *echo_states[MAX_V12PCI][MAX_PORTS];
static int	echo_ports[MAX_V12PCI][MAX_PORTS];
void *echo_malloc(int size);
void echo_free(void *mem);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,13)
static struct file_operations vpbhp_fops={

	NULL,
	NULL,
	NULL, 
	NULL, 
	NULL,
	vpbhp_ioctl,
};
#else
static struct file_operations vpbhp_fops={

	owner: THIS_MODULE,
	llseek: NULL,
	read: NULL,
	write: NULL,
	poll: NULL,
	ioctl: vpbhp_ioctl,
	open: NULL,
	release: NULL,
	fasync: NULL
};
#endif

static int 	numPCI;   	// number of valid PCI devices detected
static	int 	HW_R_FRAME[MAX_V12PCI];	// Size of data block from card HW_FIFO
static	int 	HW_W_FRAME[MAX_V12PCI];	// Size of data block to card HW_FIFO
static int 	board_type[MAX_V12PCI];	

// translated base address of PLX9052 regions
static unsigned char  *base0[MAX_V12PCI]; // 9052 config registers (mem mapped)
static u32             base1[MAX_V12PCI]; // 9052 config registers (io mapped)
static unsigned short *base2[MAX_V12PCI]; // V12PCI card memory
static void           *khook[MAX_V12PCI]; // kernel mode hook detectors
static void           *kring[MAX_V12PCI]; // kernel mode ring detectors
static void           *kldrop[MAX_V12PCI]; // kernel mode ring detectors

// intermediate kernel space buffer from transfers between V12PCI memory 
// user memory (application)
static short 	buf[SIZE_BUF];

static int 		irq[MAX_V12PCI];
static int 		irq_global=0;
static void 		*tx_fifo[MAX_V12PCI];
static void 		*rx_fifo[MAX_V12PCI];
static short 	default_tx[SIZE_FIFO/2];

// context info for each isr
static int 		boardNum[MAX_V12PCI];
static unsigned long	sigbuffer[MAX_V12PCI];

// buffer of samples to/from card
static short 		framerx[MAX_V12PCI][SIZE_FIFO];
static short 		frametx[MAX_V12PCI][SIZE_FIFO/2];
static short 		framelp[MAX_V12PCI][SIZE_FIFO/2];
static short		FrameRXptr[MAX_V12PCI];
static short		FrameTXptr[MAX_V12PCI];

// Revisions of each card
static int	rev[MAX_V12PCI];

// Party Conferencing Information
void do_conferencing(int bd);
struct party_guest {
	short board;
	short port;
};
struct party_invite {
	short party;
	short member;
};
static struct party_guest party_book[PARTY_MAX][PARTY_MAXMEM];
static struct party_invite party_people[3][12];
static short parties[PARTY_MAX];
static short party_count;

// Software Bridging
void do_soft_bridging(int bd);
struct bridge {
	short board;
	short port;
};
static struct bridge london[10][12];
static int sbridge_count=0;

// Loop back definitions
static int loop_back=0;
void do_loop_back(int bd);

static u32 duration = 0;

// Diagnostic and Debug varibles
#ifdef TF_SYNC
static int fifop =-1;
static int fifopl=-1;
//static int fsync =0;
static int fsync_resync =0;
//static int fsync_missing =0;
//static int fsync_testpattern =0;
#endif

#ifdef TDM_DATA
static int ISR_CountDown = 10;
#endif

#ifdef TDM_DATA_RAW
static short RxRAW[MAX_V12PCI][SIZE_FIFO+10];
static short RxIDX=0;
#endif

#ifdef TDM_DI
static int tdm_ch =12;
static char tdm_di[MAX_V12PCI];
static char tdm_d[MAX_V12PCI];
static char tdm_pv[MAX_V12PCI];	//previous value
#endif

/*---------------------------------------------------------------------------*\

				   FUNCTIONS
							
\*---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: init_module()
	AUTHOR...: David Rowe
	DATE.....: 22/12/01

	Called by the O/S when the module is loaded. Searches for
	valid V12PCI cards.

\*--------------------------------------------------------------------------*/

int init_module(void) {
	char 	s[4];
	struct 	pci_dev *dev = NULL;
	u32 	ret, oe, intctrl, i, waitstates;
	short 	cntrl, x;
	char  	pciipr;
	int	my_irq;
	int	y;

	printk(KERN_INFO NAME": Driver Version = %s\n",VERSION);
	major = register_chrdev(major, NAME, &vpbhp_fops);
	printk(KERN_INFO NAME": Registered Major = %d\n",major);

	numPCI = 0;

	// search for V12PCI devices on PCI bus.  Note we are actually using
	// a PLX9052, but this is functionally equiv to the 9050.
	while((dev = pci_find_device(PCI_VENDOR_ID_PLX, 
				PCI_DEVICE_ID_PLX_9050, dev)) != NULL) {

		// check that subsytem ID & Subsytem Vendor matches
		pci_read_config_dword(dev, 0x2c, (u32*)s);
		//printk("s[3] = %c 0x%02x s[2] = %c 0x%02x\n",s[3],s[3]&0xff,
		//       s[2],s[2]&0xff);

		// Do if "12VT" or " 6VT" // representing OpenSwitch 6 or 12
		if (((s[3] == '1')&&(s[2] == '2')&&(s[1] == 'V')&&(s[0] == 'T')) |
		((s[3] == ' ')&&(s[2] == '6')&&(s[1] == 'V')&&(s[0] == 'T')))
		{

			// OK, V12PCI or V6PCI found, so map address regions.....

			if((s[3] == '1')&&(s[2] == '2'))	{
				// change loop ranges for kloop etc.
				board_type[numPCI]=0;
				printk(NAME": Found V12PCI Card %d !\n", numPCI);
			}

			if((s[3] == ' ')&&(s[2] == '6'))	
			{
				// change loop ranges for kloop etc.
				board_type[numPCI]=6;
				printk(NAME": Found V6PCI Card %d !\n", numPCI);
			}

			#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,13)
			base0[numPCI] = ioremap(dev->base_address[0] &
				PCI_BASE_ADDRESS_MEM_MASK, SIZE_LCR);
			base2[numPCI] = ioremap(dev->base_address[2] &
				PCI_BASE_ADDRESS_MEM_MASK, sizeof(short)*SIZE_WD);

			// make sure no-one access's LCR via I/O space mapping
			base1[numPCI] = dev->base_address[1] & 
				PCI_BASE_ADDRESS_IO_MASK;
			ret = check_region(base1[numPCI], SIZE_LCR);
			request_region(base1[numPCI], SIZE_LCR, NAME);
			#else
			if(pci_enable_device(dev)) {
				printk(KERN_ERR "vpbhp: unable to enable PCI device\n");
				return -1;
			}

			base0[numPCI] = ioremap(dev->resource[0].start &
				PCI_BASE_ADDRESS_MEM_MASK, SIZE_LCR);
			base2[numPCI] = ioremap(dev->resource[2].start &
				PCI_BASE_ADDRESS_MEM_MASK, sizeof(short)*SIZE_WD);

			// make sure no-one access's LCR via I/O space mapping
			base1[numPCI] = dev->resource[1].start &
			PCI_BASE_ADDRESS_IO_MASK;
			#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
			ret = check_region(base1[numPCI], SIZE_LCR);
			#endif
			request_region(base1[numPCI], SIZE_LCR, NAME);
			#endif

			waitstates = readl(base0[numPCI]+0x28);
			//printk("waitstates[%d] = 0x%x\n", numPCI, waitstates);
			writel(0x40410020, base0[numPCI]+0x28);     

			// set up OE generator

			// test value for development
			// User mode overrides
			// 0xffffffff => all on
			// 0x00000000 => all off
			// Each bit (from the 32 bit word) represents a time slot
			// from the TDM bus, when the bit is on only the host can
			// write to the TDM bus.
			//oe = 0x00000005; // ports 1&3
			oe = 0xffffffff;
			//oe = 0x00001000;
			x = oe & 0xff;
			memcpy_toio(base2[numPCI]+2, &x, sizeof(short));
			x = (oe>>8) & 0xff;
			memcpy_toio(base2[numPCI]+3, &x, sizeof(short));
			x = (oe>>16) & 0xff;
			memcpy_toio(base2[numPCI]+4, &x, sizeof(short));
			x = (oe>>24) & 0xff;
			memcpy_toio(base2[numPCI]+5, &x, sizeof(short));    

			// reset HW (FIFOs and codecs)
			cntrl = 0x00;
			memcpy_toio(base2[numPCI]+6, &cntrl, sizeof(short));
			cntrl = 0x80;
			memcpy_toio(base2[numPCI]+6, &cntrl, sizeof(short));
			cntrl = 0x00;
			memcpy_toio(base2[numPCI]+6, &cntrl, sizeof(short));

			// create software FIFOs

			fifo_create(&tx_fifo[numPCI], SIZE_SW_FIFO*sizeof(short));
			fifo_create(&rx_fifo[numPCI], SIZE_SW_FIFO*sizeof(short));

			// set up default TX frame
			for(i=0; i<32; i++)
		       	{
				default_tx[i] = 0;
				#ifdef TDM_DATA
				default_tx[i] = i;
				#endif
				#ifdef TDM_DATA_RAW
				default_tx[i] = i+(i&1 ? 0x80:0);
				#endif
			}
			#ifdef TDM_DATA
			default_tx[0] = 0x55;
			#endif
			
			// initialize history buffer
			//h_tmpbuf[MAX_V12PCI][32];
			for (i=0; i<MAX_V12PCI; i++){
				h_tmpbuf[i][0]=0x100;
				for (y=1; y<32; y++){
					h_tmpbuf[i][y]=0;
				}
			}

			// test values for development
			//default_tx[0] = 0x55;
			//default_tx[31] = 0x01;

			// create rest of default frame
			for(i=32; i<SIZE_FIFO/2; i++) {
				default_tx[i] = default_tx[i-32];
			}
			//
			// test values for development
			#ifndef TDM_DATA
			default_tx[12] = 0xff;
			#endif

			printk(NAME": Setting up echo cancellor..\n");
			for(y=0;y<MAX_PORTS;y++){
				if(echo_open(&(echo_states[numPCI][y]))){
					printk(KERN_ERR "vpbhp: Unable to set up echo cancellor card[%d] port[%d]\n",numPCI,y);
					return -1;
				}
				for(i=0; i<N+ECHO_TAPS; i++) {
					echoy[numPCI][y][i] = 1.0;
				}
				for(i=0; i<N; i++) {
					echos[numPCI][y][i] = 1.0;
					echoe[numPCI][y][i] = 1.0;
				}
				echo_ports[numPCI][y]=-1;
			}

			// create hook detector
			khook[numPCI] = khook_open(base2[numPCI]);

			// create ring detector
			kring[numPCI] = kring_open(base2[numPCI]);

			// create Loop Drop detector
			kldrop[numPCI] = kldrop_open(base2[numPCI]);

			// set up interrupts (DO THIS LAST!!)
			// one int for all cards, 

			if (irq_global == 0){
				//ret = pci_read_config_byte(dev, PCI_INTERRUPT_LINE, &my_irq);
				my_irq=dev->irq;
				irq[numPCI] = my_irq;
				irq_global = my_irq;
				printk(KERN_INFO NAME": Board[%d] irq = %d\n",numPCI,(int)irq[numPCI]);
				printk(KERN_INFO NAME": Assigning IRQ [%d] for all boards\n",irq_global);
				ret = pci_read_config_byte(dev, 0x3d, &pciipr);     
				//printk("pciipr = 0x%x\n",(int)pciipr);
				boardNum[numPCI] = numPCI;
				if (irq != 0) {
					ret = request_irq(irq[numPCI], isr, 
						SA_INTERRUPT|SA_SHIRQ, NAME, &boardNum[numPCI]);
					if (ret) {
						printk(KERN_WARNING NAME": can't assign irq!\n");
						irq[numPCI] = -1;
					}
					else {
						printk(KERN_INFO NAME": Board[%d] irq %d assigned OK!\n", numPCI, irq[numPCI]);

						// enable edge triggered LINTi1
						intctrl = readl(base0[numPCI]+0x4c);
						intctrl |= 1 + (1<<1) + (1<<8);
						//intctrl = (1<<4);
						writew(intctrl,base0[numPCI]+0x4c);    

						// enable interrupt
						intctrl = readl(base0[numPCI]+0x4c);
						intctrl |= (1<<6);
						writew(intctrl,base0[numPCI]+0x4c);            
						intctrl = readl(base0[numPCI]+0x4c);
						//printk("intctrl = 0x%x\n", intctrl);
					}
				}
			}

			// setup data transfer frame size
			rev[numPCI]=card_detail(numPCI);
			if (
			((rev[numPCI]==102)&& (board_type[numPCI]==6))||
			(rev[numPCI]==111)||
			(rev[numPCI]>210))
			{
				HW_R_FRAME[numPCI]=13;
				HW_W_FRAME[numPCI]=24;
			}
			else
			{
				HW_R_FRAME[numPCI]=32;
				HW_W_FRAME[numPCI]=32;
			}

			printk(NAME ": rev=%d HW_R_FRAME=%d, HW_W_FRAME=%d\n",rev[numPCI],HW_R_FRAME[numPCI],HW_W_FRAME[numPCI]);
			// set the frame pointer to the begining of the frame
			FrameRXptr[numPCI]=0;
			FrameTXptr[numPCI]=0;
			tmpbufidx[numPCI]=0;

			holdbuf_end[numPCI]=0;
			holdbuf_lastfs[numPCI]=0;
			#ifdef TDM_DI
			tdm_di[numPCI] = 0;
			tdm_d[numPCI]= 1;
			tdm_pv[numPCI] = -1;	
			#endif

                        // Set IIC control for user i/o 2 & 3.
                        // Note: This should be the default after reset.
			cntrlbit(numPCI,IIC_SDA_IO, 0);  //Ensure =low
			cntrlbit(numPCI,IIC_SDA_DIR,0);  // i/p Hi-Z
			cntrlbit(numPCI,IIC_SDA_CTL,0);  // User I/O
			cntrlbit(numPCI,IIC_SCL_IO, 0);  //Ensure =low
			cntrlbit(numPCI,IIC_SCL_DIR,0);  // i/p Hi-Z
			cntrlbit(numPCI,IIC_SCL_CTL,0);  // User I/O
			
			numPCI++;
		}
	}

	if(numPCI==0)
		printk(NAME": ERROR: OpenSwitch card is not recognised on PCI bus\n"); 
	else
		printk(NAME": Detected %d V12PCI boards on PCI bus\n",numPCI);

	// set up bitrev lut
	for(i=0; i<ALAW_CODES; i++) {
		bitrev_lut[i] = i;
	}
	bitrev(bitrev_lut, ALAW_CODES);

	// Set up Party Conferencing stuff
	printk(NAME": Setting up conferencing...\n");
	for(x=0;x<PARTY_MAX;x++){
		parties[x]=0;
		for(y=0;y<PARTY_MAXMEM;y++){
			party_book[x][y].board=-1;
			party_book[x][y].port=-1;
		}
	}
	for(x=0;x<3;x++){
		for(y=0;y<12;y++){
			party_people[x][y].party=-1;
			party_people[x][y].member=-1;
		}
	}
	party_count=0;
	printk(NAME": Setting up soft bridging...\n");
	for(x=0;x<10;x++){
		for(y=0;y<12;y++){
			london[x][y].board=-1;
			london[x][y].port=-1;
		}
	}
	printk(NAME": Setting up alaw2linear lut...\n");
	for(y=0;y<0x100;y++){
		alaw2linear_lut[y]=alaw2linear(y);
	}
	printk(NAME": Setting up linear2alaw lut...\n");
	for(y=0;y<0x10000;y++){
		linear2alaw_lut[y]=linear2alaw(y-LUTOS);
	}
	#ifdef VPB_ASTERISK_TIMING
	printk(NAME": Setting up Asterisk timing source...\n");
	vpb_zt_init();
	#endif

	return 0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: isr()
	AUTHOR...: David Rowe
	DATE.....: 22/12/01

	Called in response to hardware interrupts approximately once every
	ms.

\*--------------------------------------------------------------------------*/

// delay measurment ---------------------------------------------

//#define DLY_MEASURE
#define DLY_IDLE    0
#define DLY_TIMING  1
#define DLY_START   100
#define DLY_TIMEOUT 10000
#define DLY_UW      0x55

static int dly_state = DLY_IDLE;
static int dly_count = 0;
static int dly_timer = 0;
static int dly_slot = 0;
static u16 dly_uw = DLY_UW;

//#define MISS_INT
#ifdef MISS_INT
static int dly_intcnt = 0;
#endif

static int dly_delay = 0;
static int dly_last_delay = 0;
static int dly_cycles = 0;
static int dly_delta = 0;

void dly_tx(short *pbuf);
void dly_rx(short *pbuf, int loop_count);

// isr entry ---------------------------------------------
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
void isr(int irq, void *dev_id, struct pt_regs *regs) 
#else
irqreturn_t isr(int irq, void *dev_id, struct pt_regs *regs)
#endif
{
	short *pbuf,*ptbuf;
	u32  intctrl;
	int  i, j, ret;
	int bd = *(int*)dev_id;

	// state variables for FPU
	unsigned long cr0;
	unsigned long linux_fpe[27];

	timestamp_t start_isr, finish_isr;
	
	//#define TIMING
	#ifdef TIMING
	u32  ctrl;
	#endif

	// check that this is infact our interupt!
	if (boardNum[bd] != bd){
		#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
		return;
		#else
		return IRQ_NONE;
		#endif
	}
	int_count++;
/*
	if ((int_count%1000) == 0){
		printk("Incomplete data count %ld \n",incomplete_frame_count);
		printk("Buffer empty count %ld \n",buffer_empty_count);
		printk("Interupt count %ld \n",int_count);
	}
*/

	// clear edge triggered LINTi1
	intctrl = readl(base0[bd]+0x4c);
	intctrl |= (1<<10);
	writew(intctrl,base0[bd]+0x4c);   

	#ifdef MISS_INT
	// used in conjunction with delay testing to test effect of
	// missing an interrupt on delay
	dly_intcnt++;
	if (dly_intcnt > 10000) {
		//if (dly_intcnt > 10005){
			dly_intcnt = 0;
		//}
		frame_count++;
	}
	else {
	#endif

	timestamp(&start_isr);

	#ifdef TIMING
	// monitor WDR node on CRO to determine time spend in ISR
	// Note this means watch dog timer cant be used during this test
        ctrl = readl(base0[bd]+0x50);
        ctrl |= (1<<5);
        writew(ctrl,base0[bd]+0x50);   
	#endif
	
	// transfer samples from user tx fifo to kernel tx buffer ------
	for (bd=0;bd<numPCI;bd++){
		pbuf = frametx[bd];
		ret = fifo_read(tx_fifo[bd], (char*)pbuf, (SIZE_FIFO/2)*sizeof(short));
		if (ret == FIFO_EMPTY) {
			pbuf = default_tx;
		}
		else {
			pbuf = frametx[bd];
		}

		/* test code that kills port 0 on OS12 */
		#ifdef DLY_MEASURE
		if ( HW_R_FRAME[bd]==13){
			dly_tx(pbuf);
		}
		#endif
	}
	bd=0; 
	#ifdef TDM_DI
	// add oscilating ramp to ch tdm_ch
//	pbuf = frametx[bd];
	for(i=0; i<8; i++)
	{
	(pbuf)[(i*32)+tdm_ch] = tdm_di[bd];
	tdm_di[bd] += tdm_d[bd];	// move value
	if(tdm_di[bd] == 127)	// check for max
		tdm_d[bd] = -1;
	if(tdm_di[bd] == -128)	// check for min
                tdm_d[bd] = 1;
	}
	#endif
	
	for (bd=0;bd<numPCI;bd++){
		if ( HW_R_FRAME[bd]==13){
			// read from RX HW FIFO
			pbuf = framerx[bd];
			read_hw_fifo(bd);
			//read_hw_fifo_13(bd);

			#ifdef TDM_DI
		if(bd==0)
		{
			// add oscilating ramp to ch tdm_ch
			pbuf = framerx[bd];
			for(i=0; i<8; i++)
			{
			j= (char)(pbuf)[(i*32)+tdm_ch] & 0xff;
		        if((j == 1+ tdm_pv[bd]) || (j == tdm_pv[bd] -1))
				;
			else
				printk("ramp mm %d\n",j);
			tdm_pv[bd] = j;	// move value
			}
		}
			#endif
			
			#ifdef TDM_DATA
			    if(ISR_CountDown >0)
			    {
				ISR_CountDown--;
				printk("\nFR");
				for(j=0; j<HW_R_FRAME[bd]; j++)
				{       printk("[%02x]%03x,",j,pbuf[j]);   }
			    }
			#endif
			
			#ifdef DLY_MEASURE
			dly_rx(framerx[bd],(SIZE_FIFO/2));
			#endif

			if(loop_back){
				do_loop_back(bd);
				pbuf = framelp[bd];

			}
			else {
				pbuf = framerx[bd];
			}
			

			// copy frames to echo buffers
			ptbuf = frametx[bd];
			for(i=0; i<(SIZE_FIFO/2); i+=32)
			{
				for(j=0; j<12; j++) {
					echos[bd][j][i/32]=(float)(alaw2linear_lut[(pbuf[i+j]&0xff)]);
					echoy[bd][j][ECHO_TAPS+(i/32)]=(float)(alaw2linear_lut[(ptbuf[i+j+12]&0xff)]);
				}
			}

			// ec

			save_cr0_and_clts(cr0);
			save_fpenv(linux_fpe);

			for (i=0;i<MAX_PORTS;i++){
				if (echo_ports[bd][i]==1){
					echo((echo_states[bd][i]), (echoe[bd][i]), &(echoy[bd][i][ECHO_TAPS]), (echos[bd][i]), 8);	
					if ( ((ECHO_SV *)(&echo_states[bd][i]))->sup == 1){
						printk(KERN_WARNING NAME": EC supression on [%d][%d]\n",bd,i);
					}
				}
			}

			restore_fpenv(linux_fpe);
			restore_cr0(cr0);
			
			/*
			for (i=0;i<8;i++){
				echoe[bd][6][i]= ec_e[i];
				echoe[bd][7][i]= ec_e2[i];
			}
			*/
			for (j=0;j<MAX_PORTS;j++){
				if (echo_ports[bd][j]==1){
					for (i=0;i<ECHO_TAPS;i++){
						echoy[bd][j][i]=echoy[bd][j][i+8];
					}
				}
			}

			// copy echo'ed buffers back to rx frames
			pbuf=framerx[bd];
			for(i=0; i<(SIZE_FIFO/2); i+=32)
			{
				for(j=0; j<12; j++) {
					if(echo_ports[bd][j]==1){
						pbuf[i+j]=linear2alaw_lut[(int)(LUTOS+echoe[bd][j][i/32])&0xffff];
					}
					else {
						pbuf[i+j]=linear2alaw_lut[(int)(LUTOS+echos[bd][j][i/32])];
					}
				}
				pbuf[i] |= 0x100; //need to put the sync bit back on!
				for(j=12;j<32;j++){
					pbuf[i+j]=0;
				}
			}

			if (party_count >0){
				do_conferencing(bd);
			}
			if (sbridge_count > 0){
				do_soft_bridging(bd);
			}

			fifo_write(rx_fifo[bd], (char*)pbuf, (SIZE_FIFO/2)*sizeof(short));
			// Write to TX FIFO.  Note FS pulse inserted here rather than in user
			// mode to ensure continuity of FS pulses even if user mode too slow
			// to fill sw FIFO.  We assume data from user mode is FS aligned.
			if (ret == FIFO_EMPTY) {
				pbuf = default_tx;
			}
			else {
				pbuf = frametx[bd];
			}
			for(i=0; i<(SIZE_FIFO/2); i+=32)
			{
				pbuf[0] |= 0x100;
				for(j=0; j<(HW_W_FRAME[bd]); j+=2) {
					memcpy_toio(base2[bd]+10, pbuf+j, 2*sizeof(short));
				}
				pbuf +=32;
			}
		}
		else {
			
			if (party_count >0){
				do_conferencing(bd);
			}
			if (sbridge_count > 0){
				do_soft_bridging(bd);
			}
			// Write to TX FIFO.  Note FS pulse inserted here rather than in user
			// mode to ensure continuity of FS pulses even if user mode too slow
			// to fill sw FIFO.  We assume data from user mode is FS aligned.

			for(i=0; i<SIZE_FIFO/2; i+=32)
			{
			    pbuf[0] |= 0x100;
		//		for(j=0; j<HW_FRAME; j++) 
				for(j=0; j<32; j++) 
				    memcpy_toio(base2[bd], pbuf+j, sizeof(short));
				pbuf +=32;
			}
			
			// transfer samples from hardware rx fifo to software rx fifo -------

			// read from RX FIFO
			
			pbuf = framerx[bd];       
			for(i=0; i<SIZE_FIFO/2; i+=32)
			{
			    for(j=0; j<32 ; j++) 
				memcpy_fromio(pbuf+j, base2[bd]+1, sizeof(short));
			    pbuf +=32;
			}

			pbuf = framerx[bd];       
			fifo_write(rx_fifo[bd], (char*)pbuf, (SIZE_FIFO/2)*sizeof(short));
			
		}
	}
	bd=0;


	// update hook and ring detector samples
	/*
	memcpy_fromio(&kldtemp[0], base2[bd]+7, 3*sizeof(short));
	sigbuffer[bd] = ((kldtemp[2] & 0x00ff)<<16) |
		         ((kldtemp[1] & 0x00ff)<<8) |
			 ( kldtemp[0] & 0x00ff);
	*/
	//printk("Sigbuffer: %06lx \n",sigbuffer[bd]);
	for (bd=0;bd<numPCI;bd++){
		khook_sample(khook[bd], board_type[bd],&sigbuffer[bd]);
		kring_sample(kring[bd], board_type[bd],&sigbuffer[bd]);
		kldrop_sample(kldrop[bd], board_type[bd],&sigbuffer[bd]);
	}
	bd=0;
		

	/*		
	save_cr0_and_clts(cr0);
	save_fpenv(linux_fpe);

	echo_unittest_iterate(8);

	restore_fpenv(linux_fpe);
	restore_cr0(cr0);
	*/
	timestamp(&finish_isr);
        timestamp_sub(&start_isr, &finish_isr);
	if (finish_isr.lo>duration){
		duration = finish_isr.lo;
	}
	
        #ifdef TIMING
	// WDR low
        ctrl = readl(base0[bd]+0x50);
        ctrl &= ~(1<<5);
        writew(ctrl,base0[bd]+0x50);            
	#endif
            	
	#ifdef MISS_INT
	}
	#endif

	#ifdef VPB_ASTERISK_TIMING
	zt_receive(&(vpb_zt.span));
	zt_transmit(&(vpb_zt.span));
	#endif

	#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	return IRQ_HANDLED;
	#endif
}


void do_loop_back(int bd){
	unsigned short *pdst = framelp[bd];
	unsigned short *psrc = frametx[bd];
	int i,j;
	
	// software loopback, rx data copy of tx data rather than
	// reading rx samples from hardware 
	// - note tx time slots offset from rx so we need to shift them
	// - note there is an introduced ERL of -20dB
	for(i=0; i<SIZE_FIFO/2; i+= 32) {
		for(j=0; j<12; j++) {
			pdst[j] = linear2alaw_lut[LUTOS+(int)(alaw2linear_lut[psrc[j+12]]*ERL_SIM) ];
		}
	    	pdst[0] |= 0x100; //need to  add sync bit has it is yet to be applied
		for(j=12;j<32;j++){
			pdst[j]=0;
		}
		pdst += 32;
		psrc += 32;
	}
}


// read data from hw fifo and place into frame buffer
void read_hw_fifo(int bd)
{
	short *pbuf,*nbuf;
	int i,j=0,buffer_empty=0,z=0,q=0;
	int pframe_sync=0;
//	int end_packet=0;
	int byte_count=0;
	short junk[1];
	int tmpbuf_total=0;
	int junked_samples=0;
	
	// Sync empty flag. (Clock into FIFO Read logic causes the R_FE flag to update)
	pbuf=junk;
	memcpy_fromio((pbuf), base2[bd]+9, sizeof(short));

	if(HW_R_FRAME[bd]==13) { // If HW. set for 13 word read per ISR
		pbuf=tmpbuf[bd];

#ifdef LONG_WORD_RX
// on entry i is available, j=0 and collects the byte count, z=0 is a tmp	
// get all data from hw fifo buffer into tmpbuf (4 words at a time)
// printk("loading pbuf\n");
		while((!buffer_empty) && ((j+4+tmpbufidx[bd])<2*SIZE_FIFO)) {
			memcpy_fromio((pbuf+j+tmpbufidx[bd]),base2[bd]+12, 4*sizeof(short));
			z=0;	// Z only goes up for each word without a Fifo empty flag set
			for(i=0; i<4; i++) {	// check each of 4 words for empty flag.
				if( *(pbuf+j+tmpbufidx[bd]+i) & 0x0200) { // if FifoEmpty set then
					// we set a flag to stop collecting words	
					buffer_empty=1;
					buffer_empty_count++;
					// we dont add this word to the input streem (no change to z)
					//printk(KERN_WARNING NAME": Hardware buffer empty bd[%d]\n",bd);
				}
				else {
					if( i != z ) { // some preceding words were empty 
						// so shift the valid ones down to fill the gaps    
						*(pbuf+j+tmpbufidx[bd]+z) = *(pbuf+j+tmpbufidx[bd]+i);
					}
					z++;
				}
			} // -for
			j += z;	// add count to the total 	
		} // -while
		byte_count=j;   // total samples this isr cycle
		if (byte_count > hwfifobuf_count_max)
			hwfifobuf_count_max = byte_count;
		if (byte_count < hwfifobuf_count_min)
			hwfifobuf_count_min = byte_count;
/*
		if (int_count%100 == 0){
			printk("Read from HW FIFO %d\n",byte_count);
		}
*/
	
		//		printk("J=%d\n",j);
		tmpbuf_total = j + tmpbufidx[bd];

		#ifdef TDM_DATA_RAW
		if(RxIDX == 0) {
			for(i=0;i<byte_count;i++)
				RxRAW[bd][10+i]= *(pbuf+i+tmpbufidx[bd]);
			RxRAW[bd][0]= byte_count;
			RxIDX = byte_count;
		}
		#endif
		
#endif

		//printk("Process pbuf\n");
		
#ifndef LONG_WORD_RX
	// the current working method for fifo data collection	
		// get all data from hw fifo buffer into tmpbuf
		while((!buffer_empty)&&(j<2*SIZE_FIFO)){
			if(z<12){
				memcpy_fromio((pbuf+j+tmpbufidx[bd]), 
						base2[bd]+12, 4*sizeof(short));
			}
			else {
				memcpy_fromio((pbuf+j+tmpbufidx[bd]), 
						base2[bd]+1, sizeof(short));
			}
			if ((pbuf+j+tmpbufidx[bd])[0]&0x0200){
					buffer_empty=1;
					//printk(KERN_WARNING NAME": Hardware buffer empty bd[%d]\n",bd);
			}
			if(z<12){
				j+=4;
				z+=4;
			}
			else {
				j++;
				z=0;
			}
		}
		byte_count=j;
		tmpbuf_total = j + tmpbufidx[bd];
		
		#ifdef TDM_DATA_RAW
		if(RxIDX == 0)
		{
		    for(i=0;i<byte_count;i++)
		        RxRAW[bd][10+i]= *(pbuf+i+tmpbufidx[bd]);
		    RxRAW[bd][0]= byte_count;
		    RxIDX = byte_count;
		}
		#endif
#endif		
		// now lets find the first frame sync
		for (i=0;i<(j+tmpbufidx[bd]);i++) {
			if(pbuf[i]&0x0100) // if we find one
			{
				pframe_sync=i;
				#ifdef TF_SYNC
				  fifop=i;
				#endif  
				i=j+tmpbufidx[bd]+10; // Set to exit loop
			}
		}
		if (i == j+tmpbufidx[bd]) {
			printk(KERN_WARNING NAME": Couldnt find first frame sync! bd[%d]\n",bd);
		}

#ifdef LONG_WORD_RX
		j=0;	// j will increment relitive to time slot number.
		z=0;
		nbuf=framerx[bd]; // always loaded with ch 0 at offset 0
		pbuf=tmpbuf[bd];
		//pbuf+=pframe_sync;	// how do we deal with this??
		// find last frame sync in tmpbuf[bd]
		q=0;
		for(i=tmpbuf_total-1; (i>0)&&(q==0); i--) {
			if(pbuf[i] & 0x100)
				q=i;
		}
		do{
			// Process slot 0
			if(j%32 == 0) {
				while ((!(pbuf[z] & 0x0100))&&(z<tmpbuf_total)){
					z++;
					junked_samples++;
				}
				// first check that there is enough data to fill a frame
				if((q>z) || (((z-1)+ HW_R_FRAME[bd])<tmpbuf_total)) {
					if(pbuf[z] & 0x0100){ // is the frame sync there?
						h_tmpbuf[bd][0] = *(nbuf+j) = pbuf[z];
						z++;
						j++;
					}
					else {
//						*(nbuf+j) = h_tmpbuf[bd][0]; // from history
						j++;
//						printk(KERN_WARNING NAME": no data, using history bd[%d]\n",bd);
					}
				}
				else {
					q= -q;	//make negetive to break out of do{}while
				}	
			}
			// Process all other slots in Rx Frame
			else if(((j%32) >0) && ((j%32) < HW_R_FRAME[bd])) {
				if(pbuf[z] & 0x0100){ // is the frame sync there?
//					*(nbuf+j) = h_tmpbuf[bd][(j%32)]; // from history
					j++;
				}
				else {
					h_tmpbuf[bd][(j%32)] = *(nbuf+j) = pbuf[z];
					z++;
					j++;
				}
			}
			// Process all slots past HW_R_FRAME[bd] frame size
			else if((j%32) >= HW_R_FRAME[bd]) {
				// Pad buffer out to 32nd slot.
				for(;(j%32)>0; j++) {
					*(nbuf+j)=0;
				}
				// on exit j will point to start of next frame (slot0)
			}
			// while we have data and have not recived too much
			//incomplete_frame_count=(incomplete_frame_count+junked_samples)/2;
			if (junked_samples > incomplete_frame_count)
				incomplete_frame_count=junked_samples;
			if (junked_samples >0)
				junked_samples_count++;
			junked_samples=0;
		}while((z < tmpbuf_total) && (j<SIZE_FIFO/2) && (q>0));
		
		// move unread words to start of tmpbuf for next isr
		j=0;
		for(i=z; i<tmpbuf_total;i++) {
			pbuf[j++] = pbuf[i];
		}
		tmpbufidx[bd]=j;	// save word count for next isr
		
#endif		
//printk("pbuf done\n");    		
#ifndef LONG_WORD_RX
                // diag. test for sync movement
		#ifdef TF_SYNC
                if( fifop != fifopl)
                    fsync_resync++;
                fifopl=fifop;
		#endif
		
		// now lets copy all valid frames to framerx
		nbuf=framerx[bd];
		pbuf=tmpbuf[bd];
		pbuf+=pframe_sync;
		for (i=0;i<SIZE_FIFO/2;i+=32){
			// check that it is a valid frame
			if (pbuf[0]&0x0100){
				for(j=0; j<HW_R_FRAME[bd]; j++) {
					*(nbuf+j)=*(pbuf+j);
				}
				// pad buffer out
				for(j=HW_R_FRAME[bd];j<32;j++) {
					*(nbuf+j)=0;
				}
			}
			else{
				break;
			}
			nbuf+=32;
			pbuf+=HW_R_FRAME[bd];
		}
		// Now lets move the unsent data to the begining of the tmpbuf
		// and adjust the tmpbuf ptr
		end_packet=HW_R_FRAME[bd]*(SIZE_FIFO/64)+pframe_sync;
		pbuf=tmpbuf[bd];
		tmpbufidx[bd]=0;
		for(i=end_packet;i<byte_count;i++){
			pbuf[tmpbufidx[bd]]=pbuf[i];
			tmpbufidx[bd]++;
		}
#endif
	}
	else {
		pbuf = framerx[bd];       
		for(i=0; i<SIZE_FIFO/2; i+=32)
		{
		    for(j=0; j<HW_R_FRAME[bd] ; j++) {
			// Copy sample from Codec to memory
			memcpy_fromio(pbuf+j, base2[bd]+1, sizeof(short));
			#ifdef TF_SYNC
			  if(*(pbuf+j) & 0x0100) fifop=j; // diag set sync pos
			#endif  
		    }
		    for(j=HW_R_FRAME[bd];j<32;j++) {
			*(pbuf+j)=0;
		    }
		    pbuf +=32;
		}
                // diag. test for sync movement
		#ifdef TF_SYNC
                if( fifop != fifopl)
                       fsync_resync++;
                fifopl=fifop;
		#endif
		
		#ifdef TDM_DATA_RAW
		if(RxIDX == 0)
		{
		    for(i=0;i<SIZE_FIFO/2;i++)
		        RxRAW[bd][10+i]= *(pbuf+i);
		        RxRAW[bd][0]= SIZE_FIFO/2;
		        RxIDX = SIZE_FIFO/2;
		}
		#endif
				
	}
	return;
}


void dly_tx(short *pbuf) {
	int   i, next_state;

	// tx side of delay measurment code, kills play audio on ch 0,
	// (first play ch) as we use this slot to measure delay
	// we loop back a single impulse thru TDM bus to determine delay

	for(i=0; i<(SIZE_FIFO/2); i+=32) {
		pbuf[i+12] = 0;
	}

	next_state = dly_state;
	if (dly_state == DLY_IDLE ) {
		dly_timer++;
		if (dly_timer == DLY_START) {
			next_state = DLY_TIMING;
			dly_count = 0;
			pbuf[12] = dly_uw;
		}
	}

	dly_state = next_state;
}


// receive side of delay measurment code

void dly_rx(short *pbuf, int loop_count) {
	int   i, next_state;
	
	next_state = dly_state;
	for(i=0; i<loop_count; i++) {
		
		if (dly_state == DLY_TIMING) {

			// look for frame sync flag
			if (*pbuf & 0x100) 
				dly_slot = 0;
			else
				dly_slot++;

			if ((*pbuf == dly_uw) && (dly_slot == 12)) {
				// found sample 
				next_state = DLY_IDLE;
				dly_timer = 0;
				dly_delay = dly_count;
				if (dly_delay != dly_last_delay) {
					dly_delta++;
				}
				dly_last_delay = dly_delay;

				dly_cycles++;
				// change unqique word every cycle
				// mod 255 bit-wise inversion
				dly_uw = (dly_uw ^ 0xff) & 0xff;
			}
			else {
				dly_count++;
				if (dly_count > DLY_TIMEOUT) {
					// time out
					next_state = DLY_IDLE;
					dly_timer = 0;
				}
		       }
		}
		dly_state = next_state;
		pbuf ++;
       }
}

	
/*--------------------------------------------------------------------------*\

	FUNCTION.: cleanup_module()
	AUTHOR...: David Rowe
	DATE.....: 15/4/00

	Called by the O/S when the module is unloaded. Frees any memory
	mapped areas for the PCI cards.

	TODO: DR 11/9/01: This function gives an error message on vfree under 
	2.4 kernel, not sure why....
	FIXED: BK 9/4/03: Because it should have been using the un-map function.

\*--------------------------------------------------------------------------*/

void cleanup_module(void) {
	int ret;
	int i,x;
	u32 intctrl;

	// Release SW fifo memory

	for(i=0; i<numPCI; i++) {
		printk("Closing down board[%d]\n",i);
		intctrl = readl(base0[i]+0x4c);
		//printk("intctrl = 0x%x\n", intctrl);

		// disable interrupts
		intctrl = readl(base0[i]+0x4c);
		intctrl &= ~(1<<6);
		writew(intctrl,base0[i]+0x4c);   
         
		if ((irq[i] != -1)&&(irq[i] != 0))
	       	{
		    free_irq(irq[i], &boardNum[i]);
		}
  
		fifo_destroy(tx_fifo[i]);
		fifo_destroy(rx_fifo[i]);

		// unmap any PCI regions
		//printk("Freeing base0[%d]\n",i);
		//vfree(base0[i]);    
		iounmap(base0[i]);    
		release_region(base1[i], SIZE_LCR);
		//printk("Freeing base2[%d]\n",i);
		//vfree(base2[i]);
		iounmap(base2[i]);
		khook_close(khook[i]);
		kring_close(kring[i]);
		kldrop_close(kldrop[i]);
		for(x=0;x<MAX_PORTS;x++){
			echo_close((echo_states[i][x]));
		}
	}

	#ifdef VPB_ASTERISK_TIMING
	vpb_zt_exit();
	#endif
    
	printk("duration = %d\n",duration);
	printk("Ignored data received %ld \n",incomplete_frame_count);
	printk("Junked count %ld \n",junked_samples_count);
//	printk("Buffer empty count %ld \n",buffer_empty_count);
	printk("Interupt count %ld \n",int_count);
//	printk("Maximum read from HW FIFO %ld \n",hwfifobuf_count_max);
//	printk("Minimum read from HW FIFO %ld \n",hwfifobuf_count_min);
/*
	printk("Maximum held in hold buffer %ld \n",holdbuf_max);
	printk("Minumum held in hold buffer %ld \n",holdbuf_min);
	printk("Number of times the hold and hw buffers under matched %ld \n",hold_hw_match_under);
	printk("Number of times the hold and hw buffers over matched %ld \n",hold_hw_match_over);
	printk("Number of times the hold and hw buffers didnt match %ld \n",hold_hw_match_not);
*/
		/*
	ret=0;
	ret = echo_unittest_evaluate();
	printk("ret = %d\n",ret);
		*/
	/*
	for (i=0; i<100; i++){
		printk("%04X\n",ben[i]);
	}
	*/
/*
	printk(KERN_INFO NAME ": Imp loc last val %d \n",imp_loc[0]);
	printk(KERN_INFO NAME ": Imp loc delta val %d \n",imp_loc_delta[0]);
	printk(KERN_INFO NAME ": Max Sample Count %d \n",max_sample_count);
	printk(KERN_INFO NAME ": Min Sample Count %d \n",min_sample_count);
	printk(KERN_INFO NAME ": Sample Count Delta %d \n",sample_count_delta);
	printk(KERN_INFO NAME ": Frame count %ld \n",frame_count);
	*/

	ret = unregister_chrdev(major, NAME);
	if (ret < 0)
		printk(KERN_WARNING "unregister_chrdev() failed!\n");

	printk(KERN_INFO "vpbhp: succesfully removed \n");
	printk("vpbhp: succesfully removed \n");
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbhp_ioctl()
	AUTHOR...: David Rowe
	DATE.....: 15/4/00

	Called when a user mode ioctl is issued, used to perform all
	PCI commands based on ioctl command.

\*--------------------------------------------------------------------------*/

static int vpbhp_ioctl(struct inode *inode,
			  struct file *filp,
			  unsigned int cmd,
			  unsigned long arg)
{
	int        ret;
	VPBHP_DATA vpbhp_data;  // ioctl parameters from user space
	int        vpbhp_addr;  // addr in card memory
	int        length;      // length of transfer in words
	short      *data;       // ptr to user space data
	int        pci_num;     // pci card number
	int        ctrl;        // value of PLX9052 control reg
	int        i;
	short      *pbuf;       // source/dest for next block
	int        howfull;
	int        howempty;
	int        khook_rd;
	int        kring_rd;
	int        kldrop_rd;
	VPBHP_FSYNC vpbhp_fsync;
	int        x,y;

	// copy parameter data from user address space

	ret = copy_from_user(&vpbhp_data, (void*)arg, sizeof(VPBHP_DATA));
	if (ret != 0)
	{
		printk(KERN_CRIT "verify area VPBHP_DATA failed\n");
		return -ENODEV;
	}

	// local copies of parameters

	vpbhp_addr = vpbhp_data.vpbhp_addr;
	if (vpbhp_addr > SIZE_WD) 
	{
		printk(KERN_WARNING "vpbhp_addr > SIZE_WD, ioctl failed!\n");    
		return -ENODEV;
	}
	length = vpbhp_data.length;
	if (length > SIZE_BUF)
	{
		printk(KERN_WARNING "length > SIZE_BUF, ioctl failed!\n");    
		return -ENODEV;
	}
	data = vpbhp_data.data;
	pci_num = vpbhp_data.pci_num;

	switch(cmd) {
		case VPBHP_IOC_PCI_SBRIDGE_ON:
			printk(KERN_INFO NAME ": Bridging %d/%d => %d/%d\n",vpbhp_addr,pci_num,((int *)data)[1],((int *)data)[0]);
			if ((london[pci_num][vpbhp_addr].board == -1)&&(london[pci_num][vpbhp_addr].port == -1)){
				// If not already bridged, increase bridge count
				sbridge_count++;
			}
			london[pci_num][vpbhp_addr].board=((int *)data)[0];
			london[pci_num][vpbhp_addr].port=((int *)data)[1];
			printk(KERN_INFO NAME ": Bridging count [%d]\n",sbridge_count);
			break;

		case VPBHP_IOC_PCI_SBRIDGE_OFF:
			printk(KERN_INFO NAME ": UN-Bridging %d/%d => %d/%d\n",vpbhp_addr,pci_num,
						london[pci_num][vpbhp_addr].port,london[pci_num][vpbhp_addr].board);
			if ((london[pci_num][vpbhp_addr].board != -1)&&(london[pci_num][vpbhp_addr].port!=-1)){
				// If not already un-bridged, decrease bridge count
				sbridge_count--;
			}
			london[pci_num][vpbhp_addr].board=-1;
			london[pci_num][vpbhp_addr].port=-1;
			printk(KERN_INFO NAME ": Bridging count [%d]\n",sbridge_count);
			break;

		case VPBHP_IOC_PCI_CONF_JOIN:
			for(x=0;x<10;x++){
				if(party_book[(short )*data][x].board == -1){
					party_book[(short )*data][x].board=pci_num;
					party_book[(short )*data][x].port=vpbhp_addr;
					party_people[pci_num][vpbhp_addr].party=(int)*data;
					party_people[pci_num][vpbhp_addr].member=x;
					x=10;
					if (parties[(int)*data] == 0){
						party_count++;
					}
					parties[(int)*data]++;
				}
			}
			break;

		case VPBHP_IOC_PCI_CONF_LEAVE:
			y=party_people[pci_num][vpbhp_addr].member;
			x=party_people[pci_num][vpbhp_addr].party;
			party_book[x][y].board=-1;
			party_book[x][y].port=-1;
			party_people[pci_num][vpbhp_addr].party=-1;
			party_people[pci_num][vpbhp_addr].member=-1;
			parties[x]--;
			if (parties[x] == 0){
				party_count--;
			}
			break;

		case VPBHP_IOC_PCI_READ_BUF_STATS:
			if (numPCI==0)
				return -ENODEV;
			pbuf = buf;
			/*
			int	max_sample_count=0;
			int	min_sample_count=1000;
			int	sample_count_delta=0;
			ulong	frame_count=0;
			ulong	incomplete_frame_count=0;
			int	imp_loc_delta=0;
			int	imp_loc=0;
			*/
			pbuf[0]=dly_state;
			pbuf[1]=dly_slot;
			pbuf[2]=frame_count;
			pbuf[3]=dly_count;
			pbuf[5]=dly_cycles;
			pbuf[7]=dly_delay;
			pbuf[8]=dly_delta;
			copy_to_user(data,buf,9*sizeof(short));
			break;

		case VPBHP_IOC_PCI_GET_NUM_CARDS:
			if (numPCI==0)
				return -ENODEV;
			copy_to_user(data, &numPCI, sizeof(int));
			break;

		case VPBHP_IOC_PCI_BLOCK_WRITE:
			if (numPCI==0)
				return -ENODEV;
			copy_from_user(buf, data, length*sizeof(short));

			// copy block of memory to single V12PCI card address

			pbuf = buf;
			for(i=0; i<length; i++) {
				memcpy_toio(base2[pci_num]+vpbhp_addr, pbuf++, sizeof(short));
			}
			break;

		case VPBHP_IOC_PCI_BLOCK_READ:
			if (numPCI==0)
				return -ENODEV;

			// copy block of memory from single V12PCI card address
			pbuf = buf;
			for(i=0; i<length; i++) {
				memcpy_fromio(pbuf++, base2[pci_num]+vpbhp_addr, sizeof(short));
			}
			copy_to_user(data, buf, length*sizeof(short));
			break;

		case VPBHP_IOC_PCI_WD_EN_L:
			if (numPCI==0)
				return -ENODEV;

			// set low PLX9052 pin user0, connected to WD enable, disabling WD timer
			ctrl = readl(base0[pci_num]+0x50);
			ctrl &= ~(1<<2);
			writew(ctrl,base0[pci_num]+0x50);            
			break;

		case VPBHP_IOC_PCI_WD_EN_H:
			if (numPCI==0)
				return -ENODEV;

			// set high PLX9052 pin user0, connected to WD enable, enabling WD
			ctrl = readl(base0[pci_num]+0x50);
			ctrl |= (1<<2);
			writew(ctrl,base0[pci_num]+0x50);            
			break;

		case VPBHP_IOC_PCI_WDR_L:
			if (numPCI==0)
				return -ENODEV;

			// set low PLX9052 pin user1, connected to WD reset, de-asserting WD reset
			ctrl = readl(base0[pci_num]+0x50);
			ctrl &= ~(1<<5);
			writew(ctrl,base0[pci_num]+0x50);            
			break;

		case VPBHP_IOC_PCI_WDR_H:
			if (numPCI==0)
				return -ENODEV;

			// set high PLX9052 pin user1, connected to WD reset, asserting WD
			// Note: WD reset is rising-edge triggered
			ctrl = readl(base0[pci_num]+0x50);
			ctrl |= (1<<5);
			writew(ctrl,base0[pci_num]+0x50);            
			break;

		case VPBHP_IOC_PCI_TX_FIFO_BLOCK_WRITE:
			if (numPCI==0)
				return -ENODEV;

			copy_from_user(buf, data, length*sizeof(short));
			fifo_write(tx_fifo[pci_num], (char*)buf, length*sizeof(short));
			break;

		case VPBHP_IOC_PCI_RX_FIFO_BLOCK_READ:
			if (numPCI==0)
				return -ENODEV;

			fifo_read(rx_fifo[pci_num], (char*)buf, length*sizeof(short));
			copy_to_user(data, buf, length*sizeof(short));
			break;

		case VPBHP_IOC_PCI_TX_FIFO_HOW_EMPTY:
			if (numPCI==0)
				return -ENODEV;
			fifo_how_full(tx_fifo[pci_num], &howfull);
			howempty = (SIZE_SW_FIFO - howfull/sizeof(short));
			copy_to_user(data, &howempty, sizeof(int));
			break;

		case VPBHP_IOC_PCI_RX_FIFO_HOW_FULL:
			if (numPCI==0)
				return -ENODEV;
			fifo_how_full(rx_fifo[pci_num], &howfull);
			howfull /= sizeof(short);
			copy_to_user(data, &howfull, sizeof(int));
			break;

		case VPBHP_IOC_PCI_READ_KHOOK:
			if (numPCI==0)
				return -ENODEV;
			khook_rd = khook_read(khook[pci_num], vpbhp_addr);
			copy_to_user(data, &khook_rd, sizeof(int));
			break;

		case VPBHP_IOC_PCI_READ_KRING:
			if (numPCI==0)
				return -ENODEV;
			kring_rd = kring_read(kring[pci_num], vpbhp_addr);
			copy_to_user(data, &kring_rd, sizeof(int));
			break;

		case VPBHP_IOC_PCI_BLOCK_EEREAD:
			if (numPCI==0)
				return -ENODEV;
			pbuf = buf;
			for(i=0; i<length; i++)
			{
				*(pbuf++) =eeread(pci_num,vpbhp_addr+i);
			}
			copy_to_user(data, buf, length*sizeof(short));
			break;

		case VPBHP_IOC_PCI_READ_KLDROP:
			if (numPCI==0)
				return -ENODEV;
			kldrop_rd = kldrop_read(kldrop[pci_num], vpbhp_addr);
			copy_to_user(data, &kldrop_rd, sizeof(int));
			break;

		case VPBHP_IOC_PCI_LOOPBACK_ON:
			if (numPCI==0)
				return -ENODEV;
			printk(NAME ": Soft Loop Back enabled on %d \n",numPCI);
			loop_back=1;
			break;

		case VPBHP_IOC_PCI_LOOPBACK_OFF:
			if (numPCI==0)
				return -ENODEV;
			printk(NAME ": Soft Loop Back disabled on %d \n",numPCI);
			loop_back=0;
			break;

		case VPBHP_IOC_PCI_HOSTECHO_ON:
			if (numPCI==0)
				return -ENODEV;
			printk(NAME ": Host Echo enabled on port %d \n",pci_num);
			x = pci_num % 12;
			y = pci_num / 12;
			echo_ports[y][x]=1;
			break;

		case VPBHP_IOC_PCI_HOSTECHO_OFF:
			if (numPCI==0)
				return -ENODEV;
			printk(NAME ": Host Echo disabled on port %d \n",pci_num);
			x = pci_num % 12;
			y = pci_num / 12;
			echo_ports[y][x]=-1;
			break;

		case VPBHP_IOC_PCI_HOSTECHO_OPT:
			if (numPCI==0)
				return -ENODEV;
//			echo_set_opt(pci_num);
			printk(NAME ": Host Echo Optimization set to %d \n",pci_num);
			break;

		#ifdef TF_SYNC
		case VPBHP_IOC_PCI_READ_FSYNC:
			/* used for debugging frame sync */
			if (numPCI==0)
				return -ENODEV;
			//vpbhp_fsync.fsync_resync = fifop;
			//vpbhp_fsync.fsync_testpattern = fifoc;
			vpbhp_fsync.fsync_resync = fsync_resync;
			//vpbhp_fsync.fsync_testpattern = fsync_testpattern;
			//vpbhp_fsync.fsync_testpattern = fsync;
			vpbhp_fsync.fsync_testpattern = fifop;
			copy_to_user(data, &vpbhp_fsync, sizeof(vpbhp_fsync));
			break;
		#endif
		
		#ifdef TDM_DATA_RAW
		// Debug Function to copy raw recived data up to user mode to allow debuging
		// of RX Fifo and associated circuits.
		// RxIDX is used to pass size and signal ISR when to do a capture of the data.
		// RxRAW is the temp buffer for data retreval.
		case VPBHP_IOC_PCI_READ_RAWTDM:
		        if (numPCI==0)
		            return -ENODEV;
		        pbuf= buf;
		        if(RxIDX > 0)
		        {
		            if((RxIDX+10) < length)
		                length=RxIDX+10;
		            pbuf = RxRAW[pci_num];
		        }
		        else
		        {
		            pbuf[0]=0;        // no data
		            length=10;        // pass only 10 words (status bytes)
		        }
		        copy_to_user(data, pbuf, length*sizeof(short));
		        RxIDX =0;
		        break;
		#endif	

		case VPBHP_IOC_PCI_BLOCK_IICWRITE:
			if (numPCI==0)
				return -ENODEV;
			copy_from_user(buf+1, data, length*sizeof(short));
			// printk("IOC %d,buf %x,len %x,addr %x\n",pci_num, buf,length,dsp_addr);
			pbuf = buf;
			*(pbuf)= Iic_Blk_Write(pci_num, buf+1, length, vpbhp_addr);
			copy_to_user(data, buf, 1*sizeof(short)); // Return status byte
			break;
			
		case VPBHP_IOC_PCI_BLOCK_IICREAD:
			if (numPCI==0)
				return -ENODEV;
			pbuf = buf;
			*(pbuf)= Iic_Blk_Read(pci_num, buf+1, length, vpbhp_addr);
			copy_to_user(data, buf, (1+length)*sizeof(short));
			break;

	}
	return 0;
}


void *echo_malloc(int size) {
	void *mem;
	mem = vmalloc(size);
	if (mem == NULL){
		printk("echo_malloc couldnt get memory (%d)\n",size);
	}
	return mem;
}

void echo_free(void *mem){
//	kfree(mem);
	vfree(mem);
}

// user defined functions required by FIFO module

void *fifo_malloc(int size) {
	return vmalloc(size);
}

void fifo_free(void *mem) {
	vfree(mem);
}

void fifo_memcpy(char *dest, char *src, int length) {
  int i;
  for(i=0; i<length; i++) {
    *dest++ = *src++;
  }
}

// user defined functions required by khook module

void *khook_malloc(int size) {
	return vmalloc(size);
}

void khook_free(void *mem) {
	vfree(mem);
}

int khook_readw(unsigned short *addr) {
	short hk;
        memcpy_fromio(&hk, addr, sizeof(short));
	return hk;
}

// user defined functions required by kring module

void *kring_malloc(int size) {
	return vmalloc(size);
}

void kring_free(void *mem) {
	vfree(mem);
}

int kring_readw(unsigned short *addr) {
	short hk;
        memcpy_fromio(&hk, addr, sizeof(short));
	return hk;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: eeread(int addr)
	AUTHOR...: David Rowe 	Updates:Peter Wintulich
	DATE.....: 5-FEB-2003

	Card Data Read function. Reads a single word from eeprom addr.
\*--------------------------------------------------------------------------*/

unsigned int eeread(int PCIn, int addr) {
	int i;
	int d;

	cntrlbit(PCIn,EEPROM_CS,1);
	cntrlbit(PCIn,EEPROM_CLK,0);

	wrbit(PCIn,1); wrbit(PCIn,1); wrbit(PCIn,0);
	for(i=0; i<6; i++)
	    wrbit(PCIn,addr>>(5-i));
	
	d = 0;
	for(i=0; i<16; i++)
       	{
	    d <<= 1;
	    d += rdbit(PCIn);
	}
        cntrlbit(PCIn,EEPROM_CS,0);
        return(d);
}
                            
/*--------------------------------------------------------------------------*\

	FUNCTION.: cntrlbit(int PCInn, int bit, int val)
	AUTHOR...: David Rowe
	DATE.....: 5-FEB-2003

	Bit # set to val, Support function for eeprom read
\*--------------------------------------------------------------------------*/

void cntrlbit(int PCInn, int bit, int val)
{
	unsigned int cntrl;

	val &= 1;
     	// first reset bit
	cntrl = readl(base0[PCInn]+0x50);
	cntrl &= ~(1<<bit);
	// now set to val
	cntrl |= val<<bit;
	writel(cntrl,base0[PCInn]+0x50);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: wrbit(int PCInn, int bit)
	AUTHOR...: David Rowe
	DATE.....: 5-FEB-2003

	Write data bit, Support function for eeprom read
\*--------------------------------------------------------------------------*/

void wrbit(int PCInn, int bit) 
{
        cntrlbit(PCInn,EEPROM_WRBIT, bit);
        cntrlbit(PCInn,EEPROM_CLK, 1);
        cntrlbit(PCInn,EEPROM_CLK, 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: rdbit(int PCInn)
	AUTHOR...: David Rowe
	DATE.....: 5-FEB-2003

	Read data bit, Support function for eeprom read

\*--------------------------------------------------------------------------*/

int rdbit(int PCInn) 
{
        unsigned int cntrl;

        cntrlbit(PCInn,EEPROM_CLK, 1);
        cntrlbit(PCInn,EEPROM_CLK, 0);
        cntrl = readl(base0[PCInn]+0x50);
        cntrl >>= EEPROM_RDBIT;
        cntrl &= 1;
        return(cntrl);
}
                               
/*--------------------------------------------------------------------------*\

	FUNCTION.: card_detail(int pci_n)
	AUTHOR...: Peter Wintulich
	DATE.....: 5-FEB-2003

	Print details of card to console & log

\*--------------------------------------------------------------------------*/

int card_detail(int pci_n)
{
	int 	myrev=0, i, x=0;
	char	st[16];
	char	hex[17]={"0123456789abcdef"};
	unsigned short eebuf[20];
	
	i=50;
	while(i<56)
	    eebuf[x++] = eeread(pci_n,i++);	

	// Decode date
	x=i=0;
	st[x++]=hex[((eebuf[i]>>12)& 0x0f)];	//day
	st[x++]=hex[((eebuf[i]>>8)& 0x0f)];
	st[x++]='/';
	st[x++]=hex[((eebuf[i]>>4)& 0x0f)];	//month
	st[x++]=hex[(eebuf[i] & 0x0f)];
	st[x++]='/';
	i++;
	st[x++]=hex[((eebuf[i]>>12)& 0x0f)];	//year
	st[x++]=hex[((eebuf[i]>>8)& 0x0f)];
	st[x++]=hex[((eebuf[i]>>4)& 0x0f)];
	st[x++]=hex[(eebuf[i] & 0x0f)];
	st[x]=0;	
	i++;
	printk(NAME ": Manufactured %s\n",st);
	
	x=0;
	st[x++]=hex[((eebuf[i]>>12)& 0x0f)];	//card
	st[x++]=hex[((eebuf[i]>>8)& 0x0f)];
	st[x++]=0;
	printk(NAME ": Card version %s.",st);
	myrev= atoi(st);
	
	x=0;
	st[x++]=hex[((eebuf[i]>>4)& 0x0f)];	//Rev.
	st[x++]=hex[(eebuf[i] & 0x0f)];
	st[x++]=0;
	i++;
	printk("%s\n",st);
	myrev=(myrev *100)+ atoi(st);
	
	i++;					// Resurved word;
	
	x=0;
	st[x++]=hex[((eebuf[i]>>12)& 0x0f)];	//SN:high
	st[x++]=hex[((eebuf[i]>>8)& 0x0f)];
	st[x++]=hex[((eebuf[i]>>4)& 0x0f)];
	st[x++]=hex[(eebuf[i] & 0x0f)];
	i++;
	st[x++]=hex[((eebuf[i]>>12)& 0x0f)];	//SN:low
	st[x++]=hex[((eebuf[i]>>8)& 0x0f)];
	st[x++]=hex[((eebuf[i]>>4)& 0x0f)];
	st[x++]=hex[(eebuf[i] & 0x0f)];
	st[x]=0;
	printk(NAME ": Serial number %s\n",st);

	return(myrev);
}	

/*--------------------------------------------------------------------------*\
    FUNCTION.: iicout(int PCI, int sig, int state)
    AUTHOR...: Peter Wintulich
    DATE.....: OCT-2003
 
    Set the pin "sig' to "state" using read-modify-write to preserve all
    other bits in this word. IIC uses open collector drives, this is being
    emulated by changing the pin direction. This is also required because
    the pin state must be readable in its open state to determin if the
    line is being driven by another device on the bus. 

	In	:PCI	Target PCI device (VPB CARD)
	                :sig    Label of signal to read:
			        IIC_SDA
				IIC_SCL
			:state	State to set pin to:
				IIC_LOW
				IIC_OPEN	
 
\*--------------------------------------------------------------------------*/
void iicout(int PCInn, int sig,int state)
{
	//printk("IIC out ==>\n");
	switch (sig)
	{
		case IIC_SDA:
		    if(state == IIC_LOW)
		    {	
		        cntrlbit(PCInn,IIC_SDA_DIR,1);  // o/p low
			cntrlbit(PCInn,IIC_SDA_IO, 0);	//Ensure =low
		    }
	    	    else
		        cntrlbit(PCInn,IIC_SDA_DIR,0);  // o/p Hi-Z	    
		    break;
                case IIC_SCL:
                    if(state == IIC_LOW)
                    {
                        cntrlbit(PCInn,IIC_SCL_DIR,1);  // o/p low
                        cntrlbit(PCInn,IIC_SCL_IO, 0);  //Ensure =low
                    }
                    else
                        cntrlbit(PCInn,IIC_SCL_DIR,0);  // o/p Hi-Z
                    break;	    
	}	    
}

/*--------------------------------------------------------------------------*\
    FUNCTION.: iicin(int PCI, int sig)
    AUTHOR...: Peter Wintulich
    DATE.....: OCT-2003

    Read the pin state of pin "sig" and return its state.
    
	In	:PCI	Target PCI device (VPB CARD)
		:sig	Label of signal to read:
				IIC_SDA
				IIC_SCL
	Return	:Bit logic level ( 0 or 1 )
\*--------------------------------------------------------------------------*/
char    iicin(int PCInn, int sig)
{
	int     tmp;
	char	cntrl=0;
		
        tmp =  readl(base0[PCInn]+0x50);	// delay
        tmp =  readl(base0[PCInn]+0x50);	// use this copy
        switch (sig)
        {
            case IIC_SDA:
		cntrl= ((tmp & (1<<IIC_SDA_IO)) ==0 ? 0:1);
		//printk("IIC in %x SDA =%s\n",tmp,(cntrl==0 ?"Lo":"Hi"));
		break;
            case IIC_SCL:
                cntrl= ((tmp & (1<<IIC_SCL_IO)) ==0 ? 0:1);
		//printk("IIC in %x SCL =%s\n",tmp,(cntrl==0 ?"Lo":"Hi"));
                break;	
	}	
        return(cntrl);
}


// user defined functions required by kldrop module

void *kldrop_malloc(int size) {
	return vmalloc(size);
}

void kldrop_free(void *mem) {
	vfree(mem);
}

int kldrop_readw(unsigned short *addr) {
	short hk;
        memcpy_fromio(&hk, addr, sizeof(short));
	return hk;
}

int alaw2linear( short alawbyte) 
{
	    unsigned long  acc,p,s,q;
	    int    sample;

	alawbyte = bitrev_lut[alawbyte];

	acc = alawbyte ^ 0x55;   /* even bit inversion */

	/* extract sign */
	if (acc & 0x80)
		p = 0x8000;
	else
		p = 0;

	acc &= 0x7f;

	/* extract q and s */
	q = acc & 0xf;
	s = acc >> 4;

	/* form linear sample */
	if (s != 0) {
		acc = 2*q+33;
		acc <<= (s-1);
	}
	else
		acc = 2*q+1;

	acc <<= 3;
	if (p)
	sample = -(short)acc;
	else
	sample = (short)acc;

	return(sample);
}

short linear2alaw(int sample)
{
	unsigned short y,mag;
	unsigned short p,s,q,b;
	unsigned long  acc;
	short alawbyte;
	acc = (sample >> 3) & 0xffff;

	/* separate sign */
	if (acc & 0x8000) {
		p = 0x80;
		mag = -(short)acc;
	}
	else {
		p = 0;
		mag = (short)acc;
	}

	/* encode magnitude */
	if (mag > 0x0fff) 
	{
		y = p + 0x7f;
	}
	else {
		/* determine left most bit and therefore segment */
		b = 0;
		acc = mag >> 1;
		while(acc) {
			b++;
			acc >>= 1;
		}

		if (b > 4) {
			s = b - 4;
			q = mag >> (b-4);
			q &= 0xf;
		}
		else {
			s = 0;
			q = mag >> 1;
		}

		y = p + (s<<4) + q;
	}

	alawbyte= y^0x55;               /* even bit inversion */

	alawbyte = bitrev_lut[alawbyte];
	return(alawbyte);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: bitrev
	AUTHOR...: David Rowe
	DATE.....: 5/6/02

	Bit reversal of samples - hardware bug work around.

\*--------------------------------------------------------------------------*/

void bitrev(unsigned char a[], int n) {
  int i,j;
  unsigned char b,c;
 
  for(i=0; i<n; i++) {
    b = a[i];
    c = 0;
    for(j=0; j<8; j++) {
      c |= ((b>>j)&0x1) << (7-j);
    }
    a[i] = c;
  }
}


/*--------------------------------------------------------------------------*\

	FUNCTION.: atoi
	AUTHOR...: Peter Wintulich
	DATE.....: 12 MAR 2003

	Ascii to Integer.

\*--------------------------------------------------------------------------*/

int atoi(char *st)
{
	int	val = 0, x=0;

	while(st[x] !=0)	// While not Null
	{
		if((st[x] >='0') && (st[x] <='9'))	//Is it in the range?
			val=(val*10)+(st[x]-'0');	// Ok add it in.
		x++;		
	}
	return (val);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: do_soft_bridging
	AUTHOR...: Ben kramer
	DATE.....: 22/05/2003

	Performs the software bridging

\*--------------------------------------------------------------------------*/

void do_soft_bridging(int bd){
	int startrx=-1;
	int bridgeBoard, bridgePort;
	int x,i;
	for(x=0;x<12;x++){ // for each port
		if((london[bd][x].board != -1)&&(london[bd][x].port != -1)){
			bridgeBoard=london[bd][x].board;
			bridgePort=london[bd][x].port;
			startrx=-1;
			for(i=0; i<32; i++) {
				if (framerx[bridgeBoard][i] & 0x100){
					startrx = i;
					i=32;
				}
			}
			if (startrx != -1) {
				for(i=0; i<SIZE_FIFO/2; i+=32) {
					frametx[bd][i+12+x] = framerx[bridgeBoard][i+startrx+bridgePort]&0xff;
				}
			}
		}
	}
}

// user defined functions required by echo module

void *malloc(int size) {
//	printk("malloc: %d\n", size);
	return vmalloc(size);
}

void free(void *mem) {
	vfree(mem);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: do_conferencing
	AUTHOR...: Ben kramer
	DATE.....: 22/05/2003

	Performs the software conferencing

\*--------------------------------------------------------------------------*/

void do_conferencing(int bd){
	int startprx[MAX_V12PCI];
	int partyIDX;
	long blah[PARTY_MAX][SIZE_FIFO/64], tmpblah;
	int x,y,i;
	y = -1;
	// Find the start frame of the fifo buffer
	for(i=0; i<32; i++) {
		for(x=0;x<numPCI;x++){
			if (framerx[x][i] & 0x100){
				startprx[x] = i;
				y=1;
			}
		}
	}
	if (y != -1) {
		// Clear the blah buffers
		for(i=0; i<SIZE_FIFO/2; i+=32) {
			for(x=0;x<10;x++){
				blah[x][i/32]=0;
			}
		}
		//  Get all the samples we need from the fifo
		// and add them up
		for(i=0; i<SIZE_FIFO/2; i+=32) {
			for (y=0;y<numPCI;y++) {
				for(x=0;x<12;x++) {
					partyIDX=party_people[y][x].party;
					if(partyIDX > -1){
						blah[partyIDX][i/32] += 
							alaw2linear_lut[framerx[y][i+startprx[y]+x]&0xff];
					}
				}
			}
		}
		// And send them back out the TX buffer
		for(i=0; i<SIZE_FIFO/2; i+=32) {
			for(x=0;x<12;x++){
				partyIDX=party_people[bd][x].party;
				if(partyIDX > -1){
					tmpblah= blah[partyIDX][i/32] - 
						alaw2linear_lut[framerx[bd][i+startprx[bd]+x]&0xff];
					if (tmpblah > 32767)  tmpblah=32767; 
					if (tmpblah < -32767) tmpblah=-32767; 
					frametx[bd][i+12+x] =  linear2alaw_lut[tmpblah+LUTOS];
				}
			}
		}
	}
}

// return time in ms since system startup
unsigned long GenerictimeGetTime(void) {
	struct timeval tval;
	do_gettimeofday(&tval);
	return ((unsigned long)(tval.tv_usec));
}


void read_hw_fifo_13(int bd){
	short	junk[1];
	short	reader[4];
	short	hwfifobuf[SIZE_HW_FIFO];	// holds data from HW FIFO
	int	hwfifobuf_count=0;		// number of words read from HW FIFO
	int	hwfifobuf_firstfs=0;		// first frame sync found in the hwfifo buf
	short	*phwfifobuf;			// pointer to hwfifobuf
	
	short	*pholdbuf=0;			// pointer to holdbuf
	short	*pframerx;			// pointer to the framerx buffer

	int	hwfifo_empty=0;			// HW FIFO empty flag

	int 	i,j;

	// Sync empty flag. (Clock into FIFO Read logic causes the R_FE flag to update)
	memcpy_fromio(junk, base2[bd]+9, sizeof(short));

	// empty HW FIFO into tmpbuf
	phwfifobuf=hwfifobuf;
	while((!hwfifo_empty)&&(hwfifobuf_count < SIZE_HW_FIFO)) {
		memcpy_fromio(&reader,base2[bd]+12, 4*sizeof(short));
		for (i=0;i<4;i++){
			if ( reader[i] & 0x0200 ){
				hwfifo_empty=1;
			}
			else {
				*(phwfifobuf)=reader[i];
				phwfifobuf++;
				hwfifobuf_count++;
			}
		}
	}
	// Collect some statistics
	if (hwfifobuf_count > hwfifobuf_count_max){
		hwfifobuf_count_max=hwfifobuf_count;
	}
	if (hwfifobuf_count < hwfifobuf_count_min){
		hwfifobuf_count_min=hwfifobuf_count;
	}
	
	// Align the data into the hold buffer.
	// check that the frame sync's match up
	for(i=0;i<hwfifobuf_count;i++){
		if (hwfifobuf[i] & 0x0100){
			hwfifobuf_firstfs=i;
			i=hwfifobuf_count;
		}
	}
	if ( ((holdbuf_lastfs[bd]+hwfifobuf_firstfs) == 13) ||((holdbuf_lastfs[bd]+hwfifobuf_firstfs) == 0)){
		pholdbuf=&holdbuf[bd][holdbuf_end[bd]];
		phwfifobuf=hwfifobuf;
	}
	else {
		if ((holdbuf_lastfs[bd]+hwfifobuf_firstfs) < 6){
			i = (holdbuf_end[bd] - holdbuf_lastfs[bd]);
			if (i<0) i=0;
			pholdbuf=&holdbuf[bd][i];
			phwfifobuf=&hwfifobuf[hwfifobuf_firstfs];
			hwfifobuf_count-=hwfifobuf_firstfs;
			// Record statistic
			hold_hw_match_under++;
		}
		else if ((holdbuf_lastfs[bd]+hwfifobuf_firstfs) > 13){
			i = (holdbuf_end[bd] - holdbuf_lastfs[bd]);
			if (i>SIZE_HW_FIFO) i=SIZE_HW_FIFO;
			pholdbuf=&holdbuf[bd][i];
			phwfifobuf=&hwfifobuf[hwfifobuf_firstfs];
			hwfifobuf_count-=hwfifobuf_firstfs;
			// Record statistic
			hold_hw_match_over++;
		}
		else {
			// fill in the difference
			if (holdbuf_lastfs[bd] < (SIZE_HW_FIFO - 13)){
			pholdbuf=&holdbuf[bd][holdbuf_lastfs[bd]];
			for (i=0;i<13;i++){
				*(pholdbuf)=0;
				pholdbuf++;
			}
			holdbuf_end[bd]=holdbuf_end[bd] - holdbuf_lastfs[bd] + 13;
			}
			phwfifobuf=&hwfifobuf[hwfifobuf_firstfs];
			hwfifobuf_count-=hwfifobuf_firstfs;
			// Record statistic
			hold_hw_match_not++;
		}
	}

	// Copy the data from the hwfifo to the hold fifo
	for (i=0;((i<hwfifobuf_count)&&(holdbuf_end[bd]<SIZE_HW_FIFO));i++){
//	for (i=0;i<hwfifobuf_count;i++){
		*(pholdbuf)=*(phwfifobuf);
		pholdbuf++;
		phwfifobuf++;
		holdbuf_end[bd]++;
	}

/*
	// Move 1 ms of samples onto the framerx buffer from the hold buffer
	pframerx=framerx[bd];
	pholdbuf=holdbuf[bd];
	if (holdbuf_end[bd] >= SIZE_FIFO/2){
		for (i=0; i<SIZE_FIFO/2; i+=32){
			for(j=0; j<HW_R_FRAME[bd]; j++) {
				*(pframerx+j)=*(pholdbuf+j);
			}
			// pad buffer out
			for(j=HW_R_FRAME[bd];j<32;j++) {
				*(pframerx+j)=0;
			}
		}
		// shuffle the hold buffer
		holdbuf_lastfs[bd]=0;
		for (i=0;i<(holdbuf_end[bd]-(SIZE_FIFO/2));i++){
			holdbuf[bd][i]=holdbuf[bd][i+(SIZE_FIFO/2)];
			// work out the last fs
			if (holdbuf[bd][i] & 0x0100){
				holdbuf_lastfs[bd]=1;
			}
			else {
				holdbuf_lastfs[bd]++;
			}
		}
		holdbuf_end[bd]-=(SIZE_FIFO/2);
	}
	else {
*/
		pframerx=framerx[bd];
		for (i=0; i<SIZE_FIFO/2; i+=32){
			j=0;
			*(pframerx+j)=0x0100;
			// pad buffer out
			for(j=1;j<32;j++) {
				*(pframerx+j)=0;
			}
		}
/*
	}

*/

	// Record statistic about how much in holdbuf
	if (holdbuf_end[bd] > holdbuf_max){
		holdbuf_max=holdbuf_end[bd];
	}
	if (holdbuf_end[bd] < holdbuf_min){
		holdbuf_min=holdbuf_end[bd];
	}

	// Done !
}

#ifdef VPB_ASTERISK_TIMING
static int vpb_zt_init(void) {
	sprintf(vpb_zt.span.name,"VPB/1");
	sprintf(vpb_zt.span.desc,"%s %d", vpb_zt.span.name, 1);
	sprintf(vpb_zt.chan.name,"VPB/%d/%d", 1, 0);
	vpb_zt.chan.chanpos = 1;
	vpb_zt.span.chans = &vpb_zt.chan;
	vpb_zt.span.channels = 0;
	vpb_zt.span.deflaw = ZT_LAW_MULAW;
	init_waitqueue_head(&vpb_zt.span.maintq);
	vpb_zt.span.pvt = &vpb_zt;
	vpb_zt.chan.pvt = &vpb_zt;
	if(zt_register(&vpb_zt.span, 0)) {
		printk(KERN_CRIT "Unable to register zaptel timing!\n");
		return -1;
	}
	return 0;
}

static int vpb_zt_exit(void) {
	if(zt_unregister(&vpb_zt.span)) {
		printk(KERN_CRIT "unable to unregister zaptel timing!\n");
		return -1;
	} else {
		printk(KERN_CRIT "unregistered zaptel timing\n");
		return 0;
	}
}
#endif


