/*--------------------------------------------------------------------------*\

    FILE....: DSPFIFO.CPP
    TYPE....: C++ Functions
    AUTHOR..: David Rowe
    DATE....: 19/11/97

    Functions used to read and write to First In First Out (FIFO) queues in
    the DSP, thus facilitating PC to DSP data transfer.  The word length of
    the FIFOs is 16 bits.

    Modified 7/10/98 to support mark 2 kernel mode device driver that
    buffers DSP fifo in PC fifo.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

        Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "dspfifo.h"
#include "hip.h"
#include "mess.h"
#include "wobbly.h"
#include "generic.h"
#include "vpbioctl.h"
#include "coff.h" 
#include "vpbreg.h"
#include "apifunc.h"
#include "mapdev.h"
/*
extern "C" {
#include "echo.h"
#include "dighyb.h"
}
*/

/*--------------------------------------------------------------------------*\

				DEFINES

\*--------------------------------------------------------------------------*/

// offsets of each param from base of DSP FIFO structure in DSP memory 

#define	PSTART		0
#define	PEND		1
#define	PWR		2
#define PRD		3
#define	SIZE		4

#define	FIFO_SIZE	5	// size of structure to upload

// error code for CheckDspFifo 

#define	FAIL		1

/*
// Host echo canc defines
#define ECHO_FIFO_SZ    640     // echo fifo size in 16-bit words
#define ECHO_N          80     // transfer block size in 16-bit words
#define IN_DELAY_FR     21       // delay between input and ref in frames
                                // This will be highly sensitive to dsp
                                // fifo size!
#define SZ_IN           (ECHO_TAPS+(IN_DELAY_FR+1)*ECHO_N)
#define IN_DELAY        (IN_DELAY_FR*ECHO_N)

// holds data for each host echo canceller thread
typedef struct {
	DspFifo *rec;
	DspFifo *play;
	int     finito;       
	void    *echo;          // handle to host echo canceller
	float   in[SZ_IN];
	int     dighyb;         // non-zero if dighyb enabled
	void    *dh;            // dig hyb states
	int     echo_en;        // non-zero if host ec enabled, useful for
                                // enabling just dig hyb, in which case we
	                        // still need echo threads running
	int     filelog_en;     // non-zero if logging play and rec signals
	                        // at input to EC to disk
	void    *wrec, *wplay;  // wave files used for logging if enabled
	int     ch;             // channel we are working with
} ECHO_THREAD;
*/

/*-------------------------------------------------------------------------*\

				CLASS

\*-------------------------------------------------------------------------*/

// pure virtual base class implementation

class baseDspFifoData {
public:	
	baseDspFifoData() {}
	virtual ~baseDspFifoData() {}
	virtual int Write(word *buf, USHORT size) = 0;
	virtual int Read(word *buf, USHORT size) = 0;
	virtual int HowFull(void) = 0;
	virtual int DspHowFull(void) = 0;
	virtual int HowEmpty(void) = 0;
};

// DspFifo implementation for original device driver

#pragma pack(1)		// force member alignment same as DSP

class DspFifoData : public baseDspFifoData {
    // copy of FIFO parameters in DSP memory 

    word	pstart;		// first word in FIFO			
    word	pend;		// one after last word in FIFO		
    word	pwr;		// write pointer			
    word	prd; 		// read pointer				
    USHORT	size;		// total storage in FIFO		

    // extra params for PC only 

    Hip		*hip;		// hip object used to talk to DSP
    USHORT	board;		// VPB board number			
    USHORT	adspfifo;	// address of structure in DSP memory	
    int		direction;	// direction of DSP FIFO

public:

	DspFifoData(Hip *hip, USHORT board, USHORT fifo_addr, int direction, int relay);
	~DspFifoData();
	int Write(word *buf, USHORT size);
	int Read(word *buf, USHORT size);
	int HowFull(void);
	int HowEmpty(void);
	int DspHowFull(void);
	int CheckDspFifo();
        void dump();
};

#pragma pack()	// normal member alignment

class DspFifoDataDD : public baseDspFifoData {
	void    *hndFile;	// handle to device driver
	USHORT	*pfifobuf;	// user mode PC fifo buffer
	USHORT	index;		// id for this fifo
	int     direction;	// direction of relay fifo
	int	size;		// available storage in PC side of relay fifo
	int	relay;		// switch for enabling relay

public:

	DspFifoDataDD(Hip *hip, USHORT board, USHORT fifo_addr, int direction, int relay);
	~DspFifoDataDD();
	int Write(word *buf, USHORT size);
	int Read(word *buf, USHORT size);
	int HowFull(void);
	int DspHowFull(void);
	int HowEmpty(void);
};

// V12PCI DSP fifo implementation

class DspFifoV12PCI: public baseDspFifoData {
	USHORT	size;		// total storage in FIFO		
	USHORT	board;		// VPB board number			
	int	direction;	// direction of DSP FIFO
	Fifo    *fifo;          // regular PC fifo used for implementation
	
public:

	DspFifoV12PCI(int direction, int size);
	~DspFifoV12PCI();
	int Write(word *buf, USHORT size);
	int Read(word *buf, USHORT size);
	int HowFull(void);
	int HowEmpty(void);
        int DspHowFull(void) {return 0;}
};

/*
   NOTE on DSP FIFO implementation:

   These FIFOs use the position of pwr and prd to determine how full the
   FIFO is.

   One way to implement a FIFO is to maintain a counter of how many words are
   in the FIFO.  The write side increments the counter, the read side
   decrements the counter.  This is a problem in our case, as the read and
   write operations are performed by unsynchronised processes (DSP and PC),
   thus the possibility exists of both sides writing to the counter at the
   same time.

   The alternative used here is to use the position of pwr and prd to
   determine how full the FIFO is.  Only the write side modifies pwr and only
   the read side modifies prd, so no contentions will occur.

   One problem with using pwr and prd is the case of pwr = prd.  This could
   occur when the FIFO is empty or when the FIFO is full.  This is resolved
   by only allowing pwr = prd when the FIFO is empty.  When writing to the
   FIFO the pwr ptr is only allowed to get up to prd-1.  Thus the actual
   capacity of the FIFO is size-1.
*/

// used to make access to DSP thread safe

static GENERIC_CRITICAL_SECTION	DspFifoSect;

// used to give each DSP fifo a unique id

static int index_count;

// size of PC side of ALL relay (signal and message) fifos

static unsigned long szRelayBuf;

/*--------------------------------------------------------------------------*\

				 FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: open_dspfifo
	AUTHOR...: David Rowe
	DATE.....: 23/3/98

	Initialises the critical section for the DSP fifo module.  
	This function should be called before any DSP fifos objects are 
	created.

\*--------------------------------------------------------------------------*/

void dspfifo_open(unsigned long szrb)
{
	GenericInitializeCriticalSection(&DspFifoSect);
	index_count = 0;
	szRelayBuf = szrb;

	// check device driver direction codes match those in header

	assert(DSPFIFO_UP == RELAY_UP);
	assert(DSPFIFO_DOWN == RELAY_DOWN);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: close_dspfifo
	AUTHOR...: David Rowe
	DATE.....: 23/3/98

	Deletes the critical section for the DSP fifo module.  This function
	should be called all DSP fifo objects have been deleted.

\*--------------------------------------------------------------------------*/

void dspfifo_close()
{
	GenericDeleteCriticalSection(&DspFifoSect);
}

/*--------------------------------------------------------------------------*\

			  DspFifo MEMBER FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::DspFifo
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Opens a link to a DSP FIFO.  Assumes DSP has been booted with 
	appropriate code.

\*--------------------------------------------------------------------------*/

DspFifo::DspFifo(Hip *h, USHORT bd, USHORT dspfifo, int direction, int amode)
//  Hip	    *h;		HIP port object 
//  USHORT  bd;		VPB board number for FIFO
//  USHORT  dspfido;	addr of FIFO structure in DSP		 
//  int	    direction;	direction of FIFO (DSPFIFO_UP or DSPFIFO_DOWN)
//  int	    mode	NORMAL || RELAY || ECHO
{
       mode = amode;

       if ((h->GetDeviceDriverModel() == 0)||(h->GetDeviceDriverModel() == 2)){
		// original driver

		/* User space host echo
		if (mode == DSPFIFO_ECHO) {
			// set up for host based echo cancelling
			fifo = new Fifo(ECHO_FIFO_SZ*sizeof(short));
		}
		*/
		d = new DspFifoData(h, bd, dspfifo, direction, mode);
		if (d == NULL)
			throw Wobbly(DSPFIFO_CANT_ALLOCATE_MEMORY);
	}
	else {
		// driver that contains HIP and DSPFIFO (old NT drvr only)

		d = new DspFifoDataDD(h, bd, dspfifo, direction, mode);
		if (d == NULL)
			throw Wobbly(DSPFIFO_CANT_ALLOCATE_MEMORY);
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::DspFifo
	AUTHOR...: David Rowe
	DATE.....: 23/04/02

	Alternate constructor for the V12PCI.  In this case we simulate the
	presence of a DSP, to maintain the same dsp fifo interface as the DSP 
	based cards.

\*--------------------------------------------------------------------------*/

DspFifo::DspFifo(int dir, int sz, int amode) {
	mode = amode;
	/* User space host echo
	if (mode == DSPFIFO_ECHO) {
		// set up for host based echo cancelling
		fifo = new Fifo(ECHO_FIFO_SZ*sizeof(short));
	}
	*/
	d = new DspFifoV12PCI(dir, sz);

	//printf("this 0x%x fifo 0x%x d 0x%x\n", (int)this, (int)fifo, (int)d);
	if (d == NULL)
		throw Wobbly(DSPFIFO_CANT_ALLOCATE_MEMORY);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::~DspFifo()
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Closes the connection between the DSP FIFO and the PC.

\*--------------------------------------------------------------------------*/

DspFifo::~DspFifo()
{ 
	/*User space host echo
	if (mode == DSPFIFO_ECHO) {
		delete fifo;
	}
	*/
	delete d;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::Write
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Writes a block of words from the PC to a DSP FIFO.  Returns OK
	for success.

\*--------------------------------------------------------------------------*/

int DspFifo::Write(word *buf, USHORT sz)
//  word    *buf;		buf of words to write to DSP FIFO 	
//  USHORT  sz;			size of buf				
{	
	int     ret;
	/* User space host echo
	if (mode == DSPFIFO_ECHO) {
		ret = fifo->Write(buf, sz);
		//printf("fifo writing: fifo addr 0x%x\n", (int)fifo);
	}
	else {
	*/
		ret = d->Write(buf,sz);
	/* User space host echo
		//printf("d writing: fifo addr 0x%x\n", (int)d);
	}
	*/

        return(ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::DSPWrite
	AUTHOR...: David Rowe
	DATE.....: 17/5/03

	Simulates DSP writing to a DSP FIFO.  Required for OpenSwitch cards
	to make wure we write to correct FIFO when using host ec.

	Returns OK for success.

\*--------------------------------------------------------------------------*/

int DspFifo::DSPWrite(word *buf, USHORT sz)
//  word    *buf;		buf of words to write to DSP FIFO 	
//  USHORT  sz;			size of buf				
{	
	int     ret;

	ret = d->Write(buf,sz);

        return(ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::Read
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Reads a block of words from a DSP FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int DspFifo::Read(word *buf, USHORT sz)
//  word    *buf;		buffer of words to read from DSP FIFO 	
//  USHORT  sz;			size of buf				
{
	int ret;

	/* User space host echo
	if (mode == DSPFIFO_ECHO) {
		ret = fifo->Read(buf, sz);
	}
	else 
	*/
		ret = d->Read(buf, sz);

	return(ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::DSPRead
	AUTHOR...: David Rowe
	DATE.....: 17/5/03

	Simulates DSP reading from a DSP FIFO.  Required for OpenSwitch cards
	to make wure we write to correct FIFO when using host ec.

	Returns OK if successful.

\*--------------------------------------------------------------------------*/

int DspFifo::DSPRead(word *buf, USHORT sz)
//  word    *buf;		buffer of words to read from DSP FIFO 	
//  USHORT  sz;			size of buf				
{
	int ret;

	ret = d->Read(buf, sz);

	return(ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::HowFull
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Returns the number of words used in the DSP FIFO.

\*--------------------------------------------------------------------------*/

int DspFifo::HowFull(void)
{
	int ret;

	/* User space host echo
	if (mode == DSPFIFO_ECHO) {
		USHORT words;
		fifo->HowFull(&words);
		ret = words;
	}
	else
	*/
		ret = d->HowFull();

	return ret;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::DspHowFull
	AUTHOR...: David Rowe
	DATE.....: 15/10/98

	Returns the number of words used in the actual DSP FIFO, not the
	relay fifo if enabled.  With old dd. reduces to HowFull

\*--------------------------------------------------------------------------*/

int DspFifo::DspHowFull(void)
{
	int ret;

	/* User space host echo
	if (mode == DSPFIFO_ECHO) {
		USHORT words;
		fifo->HowFull(&words);
		ret = words;
	}
	else
	*/
		ret = d->DspHowFull();
	
	return ret;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::HowEmpty
	AUTHOR...: David Rowe
	DATE.....: 9/10/98

	Returns the number of words free in the DSP FIFO.

\*--------------------------------------------------------------------------*/

int DspFifo::HowEmpty(void)
{
	int ret;

	/* User space host echo
	if (mode == DSPFIFO_ECHO) {
		USHORT words;
		fifo->HowFull(&words);
		ret = (ECHO_FIFO_SZ-1) - words;
	}
	else
	*/
		ret = d->HowEmpty();

	return ret;
}

/*--------------------------------------------------------------------------*\

			   DspFifoData MEMBER FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::DspFifoData
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Opens a link to a DSP FIFO.  Assumes DSP has been booted with 
	appropriate code.

\*--------------------------------------------------------------------------*/

DspFifoData::DspFifoData(Hip *h, USHORT bd, USHORT dspfifo, int dir, int relay)
//  Hip	    *h;	      HIP port object 
//  USHORT  bd;	      VPB board number for FIFO
//  USHORT  dspfido;  addr of FIFO structure in DSP		 
//  int	    dir	      unused in this implementation (apart from error checking)
//  int	    relay     unused in this implementation (apart from error checking)
{
    // validate arguments 

    assert(h != NULL);
    assert(bd < MAX_VPB);
    assert((dir == DSPFIFO_UP) || (dir == DSPFIFO_DOWN));
/* User space host echo
    assert((relay == DSPFIFO_RELAY) || (relay == DSPFIFO_NORMAL) 
	   || (relay == DSPFIFO_ECHO));
*/
    assert((relay == DSPFIFO_RELAY) || (relay == DSPFIFO_NORMAL)) ;

    // init DSP fifo member variables 

    hip = h;
    board = bd;
    adspfifo = dspfifo;
    direction = dir;
	
    // upload DSP FIFO parameter structure 

    GenericEnterCriticalSection(&DspFifoSect);
    hip->ReadDspSram(board, adspfifo, FIFO_SIZE, &pstart);
    GenericLeaveCriticalSection(&DspFifoSect);
    assert(CheckDspFifo() == OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::~DspFifoData()
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Closes the connection between the DSP FIFO and the PC.

\*--------------------------------------------------------------------------*/

DspFifoData::~DspFifoData()
{
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::Write
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Writes a block of words from the PC to a DSP FIFO.  Returns OK
	for success.

\*--------------------------------------------------------------------------*/

int DspFifoData::Write(word *buf, USHORT sz)
//  word    *buf;		buf of words to write to DSP FIFO 	
//  USHORT  sz;			size of buf				
{	
    USHORT  words_used;	// used space in FIFO			
    USHORT  words_free;	// free space in FIFO			
    USHORT  copy_first;	// size of first block move		
    USHORT  new_pwr;	// modified pwr	 	   		

    // validate arguments 

    assert(buf != NULL);
    assert(direction == DSPFIFO_DOWN);

    // update PC's copy of DSP FIFO state variables 

    GenericEnterCriticalSection(&DspFifoSect);
    hip->ReadDspSram(board, adspfifo+PWR, 1, &pwr);
    hip->ReadDspSram(board, adspfifo+PRD, 1, &prd);

    assert(CheckDspFifo() == OK);

    // determine if there is enough room in FIFO for buf 

    if (pwr >= prd)
		words_used = pwr - prd;
    if (prd > pwr)
		words_used = size - (prd - pwr);
    words_free = size - words_used - 1;

    if (words_free < sz) {
		GenericLeaveCriticalSection(&DspFifoSect);
		return(DSPFIFO_FULL);
	}

    // If buf overlaps end of linear array split into two block moves 

    if ((pwr + sz) > pend) {
		copy_first = (pend-pwr);

		hip->WriteDspSram(board, pwr, copy_first, buf);
		hip->WriteDspSram(board, pstart, sz-copy_first,&buf[copy_first]);
    }
    else
		hip->WriteDspSram(board, pwr, sz, buf);

    // increment pwr and wrap around if required 

    new_pwr = pwr + sz;
    if (new_pwr >= pend)
		pwr = pstart + (new_pwr - pend);
    else
		pwr = new_pwr;

    // copy pwr back to DSP 

    hip->WriteDspSram(board, adspfifo+PWR, 1, &pwr);
    assert(CheckDspFifo() == OK);
    GenericLeaveCriticalSection(&DspFifoSect);

    return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::Read
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Reads a block of words from a DSP FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int DspFifoData::Read(word *buf, USHORT sz)
//  word    *buf;		buffer of words to read from DSP FIFO 	
//  USHORT  sz;			size of buf				
{
    USHORT  words_used;	// used space in FIFO			
    USHORT  copy_first;	// size of first block move		
    USHORT  new_prd;	// modified prd	    		

    // validate arguments 

    assert(buf != NULL);
    assert(direction == DSPFIFO_UP);

    // update PC's copy of DSP FIFO state variables 

    GenericEnterCriticalSection(&DspFifoSect);
    hip->ReadDspSram(board, adspfifo+PWR, 1, &pwr);
    hip->ReadDspSram(board, adspfifo+PRD, 1, &prd);
    
    /*  Used for testing bug where SCP was interfering with PC reads.
	DR 12/2/01
	while(1) {     
		word pwr2;
		hip->ReadDspSram(board, adspfifo+PWR, 1, &pwr2);
		if (pwr2 != pwr)
			printf("pwr 0x%x  pwr2 0x%x\n",pwr,pwr2);
	}
    */
    assert(CheckDspFifo() == OK);

    // determine if there is data in FIFO to fill buf 

    if (pwr >= prd)
		words_used = pwr - prd;
    if (prd > pwr)
		words_used = size - (prd - pwr);
    if (words_used < sz) {
		GenericLeaveCriticalSection(&DspFifoSect);
		return(DSPFIFO_EMPTY);
	}

    // If buf overlaps end of linear array split into two block moves 

    if ((prd + sz) > pend) {
		copy_first = (pend-prd);

		hip->ReadDspSram(board, prd, copy_first, buf);
		hip->ReadDspSram(board, pstart, sz-copy_first,&buf[copy_first]);
    }
    else
		hip->ReadDspSram(board, prd, sz, buf);

    // increment prd and wrap around if required 

    new_prd = prd + sz;
    if (new_prd >= pend)
		prd = pstart + (new_prd - pend);
    else
		prd = new_prd;

    // copy prd back to DSP 


    hip->WriteDspSram(board, adspfifo+PRD, 1l, &prd);
    assert(CheckDspFifo() == OK);

    GenericLeaveCriticalSection(&DspFifoSect);
    return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::HowFull
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Returns the number of words used in the DSP FIFO.

\*--------------------------------------------------------------------------*/

int DspFifoData::HowFull(void)
{
    USHORT words;

    GenericEnterCriticalSection(&DspFifoSect);
    hip->ReadDspSram(board, adspfifo+PWR, 1, &pwr);
    hip->ReadDspSram(board, adspfifo+PRD, 1, &prd);
	
    assert(CheckDspFifo() == OK);
    if (pwr >= prd)
		words = pwr - prd;
    if (prd > pwr)
		words = size - (prd - pwr);

    GenericLeaveCriticalSection(&DspFifoSect);
    return(words);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::DspHowFull
	AUTHOR...: David Rowe
	DATE.....: 15/10/98

	Returns the number of words used in the actual DSP FIFO, not the
	relay fifo if enabled.  With old dd. reduces to HowFull

\*--------------------------------------------------------------------------*/

int DspFifoData::DspHowFull(void)
{
    return(HowFull());
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::HowEmpty
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Returns the number of words free in the DSP FIFO.

\*--------------------------------------------------------------------------*/

int DspFifoData::HowEmpty(void)
{
	return(size-1-HowFull());
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: DspFifoData::CheckDspFifo
    AUTHOR...: David Rowe
	DATE.....: 20/11/97

    Performs a few sanity checks on DSP FIFO structure.  Returns OK is DSP
    FIFO checks out returns.

\*-------------------------------------------------------------------------*/

int DspFifoData::CheckDspFifo()
{
  hip->ReadDspSram(board, adspfifo+PSTART, 1, &pstart);
  hip->ReadDspSram(board, adspfifo+PEND, 1, &pend);
    if (pend < pstart) dump();
    if (pwr < pstart) dump();
    if (pwr > pend) dump();
    if (prd < pstart) dump();
    if (prd > pend) dump();

    return(OK);
}

void DspFifoData::dump() {
  printf("\npstart = 0x%x\n",pstart);
  printf("pend = 0x%x\n",pend);
  printf("pwr = 0x%x\n",pwr);
  printf("prd = 0x%x\n",prd);

  // read again - just to be sure
  hip->ReadDspSram(board, adspfifo+PWR, 1, &pwr);
  hip->ReadDspSram(board, adspfifo+PRD, 1, &prd);
  hip->ReadDspSram(board, adspfifo+PSTART, 1, &pstart);
  hip->ReadDspSram(board, adspfifo+PEND, 1, &pend);
  printf("\npstart = 0x%x\n",pstart);
  printf("pend = 0x%x\n",pend);
  printf("pwr = 0x%x\n",pwr);
  printf("prd = 0x%x\n",prd);

  //coff_check_dsp_firmware(hip, 0, "/etc/vpb/vpbmain.out");
  assert(0);
}

/*--------------------------------------------------------------------------*\

			  DspFifoDataDD MEMBER FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::DspFifoDataDD
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Opens a link to a relay FIFO.  Assumes DSP has been booted with 
	appropriate code.  A relay fifo is a large PC fifo that buffers
	communications to the underlying DSP fifo.  A device driver thread
	then transfers data from the PC to the DSP fifo (or vice-versa
	depending on direction of relay).

	Optionally, (relay flag == RELAY_OFF), we can just talk to the DSP
	FIFO direct, without the PC relay.  This would found to be necessary
	for the up and downmessage queues as the large buffers were causing
	messages to be segmented (length separated from body of mess), which
	caused all sorts of nasty problems.  Big buffers are only required
	for high bandwidth signal data anayway.

\*--------------------------------------------------------------------------*/

DspFifoDataDD::DspFifoDataDD(Hip *h, USHORT bd, USHORT dspfifo, int dir, int rel)
//  Hip		*h;			HIP port object 
//  USHORT  bd;			VPB board number for FIFO
//  USHORT  dspfido;	addr of FIFO structure in DSP		 
//  int		dir;		direction of relay (DSPFIFO_UP, DSPFIFO_DOWN)
//	int		rel			relay flag
{
    assert(0); // obsolete code - should never be called
    #ifdef OLD_CODE

	long			IoctlCode;
    unsigned long	ReturnedLength; 
    int				IoctlResult;
	FIFO_PARAMS		Fifo;

	GenericEnterCriticalSection(&DspFifoSect);

    // validate arguments 

    assert(h != NULL);
	assert(bd < MAX_VPB);
	assert((dir == DSPFIFO_UP) || (dir == DSPFIFO_DOWN));
	assert((rel == RELAY_ON) || (rel == RELAY_OFF));

	// get handle to device driver from hip

	hndFile = h->GethndFile();
	index = index_count++;
	direction = dir;
	size = szRelayBuf;
	relay = rel;

	if (relay == RELAY_ON) {

		// allocate user mode buffer for PC side of buffer
	
		pfifobuf = new USHORT[szRelayBuf];
		assert(pfifobuf != NULL);
	
		// set up DeviceIoControl() call

		Fifo.ppc = (void*)pfifobuf;
		Fifo.length = szRelayBuf;
		Fifo.index = index;
		Fifo.board = bd;
		Fifo.pdsp = dspfifo;
		Fifo.direction = direction;

	    IoctlCode = IOCTL_VPB_RELAY_FIFO_CREATE;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            NULL,		        // Buffer from driver.
                            0,			        // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}
	else {
		Fifo.index = index;
		Fifo.pdsp = dspfifo;
		Fifo.board = bd;

		//		IoctlCode = IOCTL_VPB_DSP_FIFO_CREATE;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            NULL,		        // Buffer from driver.
                            0,			        // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}

	GenericLeaveCriticalSection(&DspFifoSect);
	if (!IoctlResult)                            // Did the IOCTL succeed?
        throw Wobbly(DSPFIFO_OPEN_FAIL);	
	#endif
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::~DspFifoDataDD()
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Closes the connection between the relay FIFO and the PC.

\*--------------------------------------------------------------------------*/

DspFifoDataDD::~DspFifoDataDD()
{
	assert(0); // obsolete code - should never be called
	#ifdef OLD_CODE

	long			IoctlCode;
    unsigned long	ReturnedLength; 
    int				IoctlResult;
	FIFO_PARAMS		Fifo;
	
	GenericEnterCriticalSection(&DspFifoSect);
	Fifo.index = index;
	
	if (relay == RELAY_ON) {

		// set up DeviceIoControl() call
		
		//		IoctlCode = IOCTL_VPB_RELAY_FIFO_DESTROY;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(Fifo),		// Length of buffer in bytes.
                            NULL,		        // Buffer from driver.
                            0,			        // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
		delete [] pfifobuf;
	}
	else {
		// set up DeviceIoControl() call

		//		IoctlCode = IOCTL_VPB_DSP_FIFO_DESTROY;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(Fifo),		// Length of buffer in bytes.
                            NULL,		        // Buffer from driver.
                            0,			        // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}

	GenericLeaveCriticalSection(&DspFifoSect);
	if (!IoctlResult)                            // Did the IOCTL succeed?
        throw Wobbly(DSPFIFO_CLOSE_FAIL);	

	#endif
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::Write
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Writes a block of words from the PC to a relay FIFO.  Returns OK
	for success.

\*--------------------------------------------------------------------------*/

int DspFifoDataDD::Write(word *buf, USHORT sz)
//  word    *buf;		buf of words to write to DSP FIFO 	
//  USHORT  sz;			size of buf				
{	
	assert(0); // obsolete code - should never be called
	#ifdef OLD_CODE

	long			IoctlCode;
    unsigned long	ReturnedLength; 
    int				IoctlResult;
	FIFO_PARAMS		Fifo;
	
	assert(direction == DSPFIFO_DOWN);

	GenericEnterCriticalSection(&DspFifoSect);
	if (relay == RELAY_ON) {

		Fifo.ppc = buf;
		Fifo.length = sz;
		Fifo.index = index;
	
		// set up DeviceIoControl() call

		//		IoctlCode = IOCTL_VPB_FIFO_WRITE;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(Fifo),		// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(Fifo),       // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	} 
	else {
		Fifo.index = index;
		Fifo.ppc = buf;
		Fifo.length = sz;

		//		IoctlCode = IOCTL_VPB_DSP_FIFO_WRITE;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}

	GenericLeaveCriticalSection(&DspFifoSect);
	if (!IoctlResult)                            // Did the IOCTL succeed?
        throw Wobbly(DSPFIFO_WRITE_FAIL);	
	if (ReturnedLength != sizeof(Fifo)) 
        throw Wobbly(DSPFIFO_WRITE_FAIL);	

	return(Fifo.ret);
    #endif
	return 0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::Read
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Reads a block of words from a relay FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int DspFifoDataDD::Read(word *buf, USHORT sz)
//  word    *buf;		buffer of words to read from DSP FIFO 	
//  USHORT  sz;			size of buf				
{
	assert(0); // obsolete code - should never be called
	#ifdef OLD_CODE

	long			IoctlCode;
    unsigned long	ReturnedLength; 
    int				IoctlResult;
	FIFO_PARAMS		Fifo;
	
	assert(direction == DSPFIFO_UP);

	GenericEnterCriticalSection(&DspFifoSect);


	if (relay == RELAY_ON) {

		Fifo.ppc = buf;
		Fifo.length = sz;
		Fifo.index = index;
	
		// set up DeviceIoControl() call

		//		IoctlCode = IOCTL_VPB_FIFO_READ;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(Fifo),		// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(Fifo),       // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}
	else {
		Fifo.index = index;
		Fifo.ppc = buf;
		Fifo.length = sz;

		//	    IoctlCode = IOCTL_VPB_DSP_FIFO_READ;
	    IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}


	GenericLeaveCriticalSection(&DspFifoSect);
	if (!IoctlResult)                            // Did the IOCTL succeed?
        throw Wobbly(DSPFIFO_READ_FAIL);	
	if (ReturnedLength != sizeof(Fifo)) 
        throw Wobbly(DSPFIFO_READ_FAIL);	

	return(Fifo.ret);
	#endif
	return 0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::HowFull
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Returns the number of words used in the PC side of the relay FIFO.

\*--------------------------------------------------------------------------*/

int DspFifoDataDD::HowFull(void)
{
	assert(0); // obsolete code - should never be called
	#ifdef OLD_CODE

	long			IoctlCode;
    unsigned long	ReturnedLength; 
    int				IoctlResult;
	FIFO_PARAMS		Fifo;
	
	GenericEnterCriticalSection(&DspFifoSect);


	if (relay == RELAY_ON) {
		Fifo.index = index;
	
		// set up DeviceIoControl() call

		//		IoctlCode = IOCTL_VPB_FIFO_HOW_FULL;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(Fifo),		// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(Fifo),       // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	
	}
	else {
		Fifo.index = index;

		//		IoctlCode = IOCTL_VPB_DSP_FIFO_HOW_FULL;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}

	GenericLeaveCriticalSection(&DspFifoSect);
	if (!IoctlResult)                           // Did the IOCTL succeed?
        throw Wobbly(DSPFIFO_HOW_FULL_FAIL);	
	if (ReturnedLength != sizeof(Fifo)) 
        throw Wobbly(DSPFIFO_HOW_FULL_FAIL);	

	return(Fifo.ret);
	#endif
	return 0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::DspHowFull
	AUTHOR...: David Rowe
	DATE.....: 15/10/98

	Returns the number of words used in the actual DSP FIFO.  Used by
	play to determine when DSP has really finished.

\*--------------------------------------------------------------------------*/

int DspFifoDataDD::DspHowFull(void)
{
	assert(0); // obsolete code - should never be called
	#ifdef OLD_CODE

	long			IoctlCode;
    unsigned long	ReturnedLength; 
    int				IoctlResult;
	FIFO_PARAMS		Fifo;
	
	GenericEnterCriticalSection(&DspFifoSect);


	Fifo.index = index;

	//	IoctlCode = IOCTL_VPB_DSP_FIFO_HOW_FULL;
	IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );

	GenericLeaveCriticalSection(&DspFifoSect);
	if (!IoctlResult)                           // Did the IOCTL succeed?
        throw Wobbly(DSPFIFO_HOW_FULL_FAIL);	
	if (ReturnedLength != sizeof(Fifo)) 
        throw Wobbly(DSPFIFO_HOW_FULL_FAIL);	


	return(Fifo.ret);
	#endif
	return 0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::HowEmpty
	AUTHOR...: David Rowe
	DATE.....: 9/10/98

	Returns the number of words available in the PC side of the relay FIFO.

\*--------------------------------------------------------------------------*/

int DspFifoDataDD::HowEmpty(void)
{
	return((size-1) - HowFull());
}

/*--------------------------------------------------------------------------*\

			  DspFifoV12PCI MEMBER FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoV12PCI
	AUTHOR...: David Rowe
	DATE.....: 23/4/02

\*--------------------------------------------------------------------------*/

DspFifoV12PCI::DspFifoV12PCI(int dir, int sz)
{
	GenericEnterCriticalSection(&DspFifoSect);

	// validate arguments 

	assert((dir == DSPFIFO_UP) || (dir == DSPFIFO_DOWN));

	size = sz;
	direction = dir;
	fifo = new Fifo(sz);

	GenericLeaveCriticalSection(&DspFifoSect);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoV12PCI::~DspFifoV12PCI()
	AUTHOR...: David Rowe
	DATE.....: 23/4/02

\*--------------------------------------------------------------------------*/

DspFifoV12PCI::~DspFifoV12PCI()
{
	GenericEnterCriticalSection(&DspFifoSect);
	delete fifo;
	GenericLeaveCriticalSection(&DspFifoSect);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoV12PCI::Write
	AUTHOR...: David Rowe
	DATE.....: 23/4/02

	Writes a block of words from the PC to a relay FIFO.  Returns OK
	for success.

\*--------------------------------------------------------------------------*/

int DspFifoV12PCI::Write(word *buf, USHORT sz)
//  word    *buf;		buf of words to write to DSP FIFO 	
//  USHORT  sz;			size of buf				
{	

	GenericEnterCriticalSection(&DspFifoSect);
	//printf("fifo 0x%x write buf[0] = %d buf[1] = %d\n", (int)this, buf[0],buf[1]); 
	int ret  = fifo->Write(buf, sz);
	GenericLeaveCriticalSection(&DspFifoSect);

	return(ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoV12PCI::Read
	AUTHOR...: David Rowe
	DATE.....: 23/4/02

	Reads a block of words from a relay FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int DspFifoV12PCI::Read(word *buf, USHORT sz)
//  word    *buf;		buffer of words to read from DSP FIFO 	
//  USHORT  sz;			size of buf				
{

	GenericEnterCriticalSection(&DspFifoSect);
	int ret = fifo->Read(buf, sz);
	GenericLeaveCriticalSection(&DspFifoSect);
	return(ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoV12PCI::HowFull
	AUTHOR...: David Rowe
	DATE.....: 23/4/02

	Returns the number of words used in the PC side of the relay FIFO.

\*--------------------------------------------------------------------------*/

int DspFifoV12PCI::HowFull(void)
{
	GenericEnterCriticalSection(&DspFifoSect);
	USHORT words;
	fifo->HowFull(&words); 
	GenericLeaveCriticalSection(&DspFifoSect);
	return(words);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoV12PCI::HowEmpty
	AUTHOR...: David Rowe
	DATE.....: 23/4/02

	Returns the number of words available in the PC side of the relay FIFO.

\*--------------------------------------------------------------------------*/

int DspFifoV12PCI::HowEmpty(void)
{
	return((size-1) - HowFull());
}

/*--------------------------------------------------------------------------*\

			  HOST ECHO CANCELLER FUNCTIONS

\*--------------------------------------------------------------------------*/

#ifdef LINUX_USER_ECHO
// DR 12/2/03 host echo canc not supported under Win32 just yet.......

/*--------------------------------------------------------------------------*\

	FUNCTION.: dspfifo_echo_start_thread
	AUTHOR...: David Rowe
	DATE.....: 24/9/02

	This function starts the host ec thread running.  It must be called
	once for each thread, after the two dsp fifos for each ch have been
	created.

\*--------------------------------------------------------------------------*/

void *dspfifo_echo_start_thread(int bd, int ch) {
	ECHO_THREAD *ec = new ECHO_THREAD;
	VPBREG      *v;   
	DspFifo     *rec;
	DspFifo     *play;
	int         sample, last_sample, hasnt_changed;
        int         i;

	ec->ch = ch;

	// init logging to file

	ec->filelog_en = 1; // DR 16/5/03 - control at compile time for now

	if (ec->filelog_en) {
		char fn[VPB_MAX_STR];
		sprintf(fn, "rec%d.wav", ch);
		vpb_wave_open_write(&ec->wrec, fn, VPB_LINEAR);
		sprintf(fn, "play%d.wav", ch);
		vpb_wave_open_write(&ec->wplay, fn, VPB_LINEAR);
	}

	v = vpb_c->vpbreg(bd);
	rec = v->rxdf[ch];
	play = v->txdf[ch];

	// make sure record fifo has filled before continuing, so that
	// delay between record and play side is known.  Note that this
	// code is likely to be quite sensitive (fragile) to changes in
	// the fifo size (set in vpbreg.cpp)

	// as size of dsp fifo may vary (set in vpbreg), and it also
	// depends on the number of samples in each frame (e.g. 160)
	// use an algorthim that waits for read fifo to stop changing
	// to determine when it is full

	hasnt_changed = 0;
	last_sample = 0;
	do {
		sample = rec->d->HowFull();
		if (sample == last_sample)
			hasnt_changed++;
		else
			hasnt_changed = 0;
		GenericSleep(20);
		last_sample = sample;
	} while(hasnt_changed < 5);
	mprintf("rec->d->HowFull() = %d\n", sample);
		sample = play->d->HowFull();
	mprintf("play->d->HowFull() = %d\n", sample);

	ec->play = play;
	ec->rec = rec;
	ec->finito = 0;
	ec->dighyb = v->dighyb;
	ec->echo_en = v->echo_en;
	mprintf("echo_en = %d dighyb = %d\n", ec->echo_en,ec->dighyb);

	if (ec->dighyb) {
		char hyb_file[VPB_MAX_STR];

		sprintf(hyb_file,"%s%d",v->dighybfile,mapdevtohndle(bd,ch));
		mprintf("opening hybrid file %s\n",hyb_file);
		int ret = dighyb_open(&ec->dh, hyb_file);

		// if this fires probably bad path to hybrid file
		assert(ret == 0);
	}

	echo_open(&ec->echo);
	for(i=0; i<SZ_IN; i++)
		ec->in[i] = 0;

	Generic_beginthread(dspfifo_echo_thread, 0, (void*)ec);

	return (void*)ec;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: dspfifo_echo_stop_thread
	AUTHOR...: David Rowe
	DATE.....: 24/9/02

	Shut down function for ec thread.

\*--------------------------------------------------------------------------*/

void dspfifo_echo_stop_thread(void *pv) {
	ECHO_THREAD *ec = (ECHO_THREAD*)pv;

	ec->finito = 1;
	while(ec->finito)
		GenericSleep(5);

	echo_close(ec->echo);
	if (ec->dighyb)
		dighyb_close(ec->dh);

	if (ec->filelog_en) {
		vpb_wave_close_write(ec->wrec);
		vpb_wave_close_write(ec->wplay);
	}

	delete ec;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: dspfifo_echo_thread
	AUTHOR...: David Rowe
	DATE.....: 24/9/02

	This function runs as a thread:
	- it forms a layer between the dspfifo interface and the actual dsp
	fifos.
	- calling functions think they are just talking to a regular dspfifo,
	their interfaces dont change
	- however this layer performs echo cancellation
	- a thread is used for this processing so that the DSP fifo reads 
	(rec) and writes (play) at the same time.  This is required so that
	we can sync the input to reference signals for the EC.
	- one thread handles each port (which has a play and rec fifo).

\*--------------------------------------------------------------------------*/

void dspfifo_echo_thread(void *data) {
	ECHO_THREAD *ec     = (ECHO_THREAD*)data;
	DspFifo     *rec    = ec->rec; 
	DspFifo     *play   = ec->play;
	int         ret;
	word        playbuf[ECHO_N], recbuf[ECHO_N];
	float       *in     = ec->in;
	float       ref[ECHO_N], out[ECHO_N];
	int         i;

	while(!ec->finito) {
		// play side
		
		#ifdef TMP
		memset(playbuf, 0 , sizeof(playbuf));
		ret = play->fifo->Read(playbuf, ECHO_N);
		while(play->d->Write(playbuf, ECHO_N) != OK) {
			GenericSleep(1);
		}
		#endif

		// record side

 		//printf("ec thread reading [%02d] 0x%x\n", ec->ch, (int)rec->d);
     		while(rec->d->Read(recbuf, ECHO_N) != OK) {
			GenericSleep(1);
		}
		
		memset(playbuf, 0 , sizeof(playbuf));
		ret = play->fifo->Read(playbuf, ECHO_N);
		play->d->Write(playbuf, ECHO_N);

		// optional logging to file

		if (ec->filelog_en) {
			// log reference signal for ec
			vpb_wave_write(ec->wrec, (char*)recbuf, sizeof(recbuf));
		}

		#ifdef FIFO_HOWFULL
		// useful for testing if fifo is emptying, or play-rec
		// delay is changing which can kill ec
		mprintf("rec = %3d  play = %3d\n",
			       rec->d->HowFull(),
			       play->d->HowFull());
		#endif

		for(i=0; i<ECHO_N; i++) {
			// note: convert unsigned word to signed short
			in[ECHO_TAPS+IN_DELAY+i] = (float)((short)playbuf[i]);
			ref[i] = (float)((short)recbuf[i]);
		}
		if (ec->dighyb)
			dighyb(ec->dh, ref, &in[ECHO_TAPS], ref, ECHO_N);
		if (ec->echo_en) {
		    	echo(ec->echo, out, &in[ECHO_TAPS], ref, ECHO_N);
			for(i=0; i<ECHO_N; i++)
				ref[i] = out[i];
		}
		
		if (ec->filelog_en) {
			short tmp[ECHO_N];

			// optional logging of ec input signal
			for(i=0; i<ECHO_N; i++)
				tmp[i] = (short)in[ECHO_TAPS+i];
			vpb_wave_write(ec->wplay, (char*)tmp, sizeof(tmp));
		}

		// update memory
		for(i=0; i<ECHO_TAPS+IN_DELAY; i++)
			in[i] = in[i+ECHO_N];
		
		for(i=0; i<ECHO_N; i++)
			recbuf[i] = (short)ref[i];
		ret = rec->fifo->Write(recbuf, ECHO_N);
	}

	ec->finito = 0;
	Generic_endthread();
}

#endif








