/*---------------------------------------------------------------------------*\

	FILE....: cidg.cpp
	TYPE....: C Program
	AUTHOR..: Peter Wintulich
	DATE....: 12/11/02

	Module to generate caller ID signals.

	CID_EMPTY this is not a field. if used in vpb_cid_set() as the field
       	type it will initalise the structure with default values for an cid 
	string with no numbers or text, only a valid date time.
	
	CID_DATE_TIME field, the date and time stored in this field is
       	dependent on the message type. If CID_CALL_TYPE value 
	<128 then it will be current time, 
	==129 date time of recived or transfered meaasge.

      	CID_CALLING_LINE_DN, CID_CALLED_DN fields, 18 digits maximum. May
	include ' ' and '-'. If part number only delivered then last digit
       	will be '-' to signify that it is incomplete.

	CID_RFA_CLDN, CID_RFA_CN fields, 'O'== Number Unavailable
	'P'== Number withheld

	CID_CALLER_NAME and CID_TEXT_PARAMETER field are the same field. 
	If the Caller name is presented then the RFA_CN is not sent. The 
	CID_TEST_PARAMETER is intended to assist in identification of the 
	call type and may contain "Payphone", "International", 
	"Ring-back Call". Caller name should take priority if available.
	
\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <time.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "../src/vpbapi.h"

/*-----------------------------------------------------------------------\
 * 									
 *			LOCAL FUNCTION PROTOTYPES			
 * 									
\-----------------------------------------------------------------------*/
static int byte_encode(char dlpb, int *scount, short *buff, double *stime, int *bt);

static int bit_encode(int bit, int *samples_count, short *, double *Acc, 
		      int *bt);

static  int isinstr(char, char*);

static int	COUNTRY			= VPB_COUNTRY_AU;

static int 		HAMP=6300;
static int 		LAMP=6300;

/*-----------------------------------------------------------------------*\

	FUNCTION: vpb_cid_set()
	AUTHOR..: Peter Wintulich
	DATE....: Nov 2002

	VPBAPI function to load valid data of 'type' and 'value' into the
	passed CID structure.  Loading some fields will set associated
	fields to NULL automaticly as some fiedls are exclusive.
 
	Return:	Error if invalid type of value out of range, else OK

\*-----------------------------------------------------------------------*/

int WINAPI vpb_cid_set(VPB_CID *cid_struct, int type, void *value)
{	//	*cid		Structure containing cid fields
	//	type		cid Field label
	//	*value		Pointer to value
	
	int	x;
	int	len;
	int	valid;

	VPB_CID	*cid;
	cid	=cid_struct;
	
  if(cid != NULL)	// check for valid
  {
    switch (type)			
    {
	case VPB_CID_EMPTY:		// Initalize all fields 
	    strcpy(cid->date_time, "AUTO"); // time stamp inserted by compose
    	    cid->call_type   = 0;
	    cid->cldn[0]     = 0;
	    cid->cdn[0]      = 0;
	    cid->ra_cldn[0]  = 'O';	// Number unavailable
	    cid->ra_cldn[1]  = 0;
	    cid->cn[0]     = 0;
	    cid->ra_cn[0]  = 'O';	// Text/Name unavailable
	    cid->ra_cn[1]  = 0;
	    cid->nmss = 0;
	    break;
	    
	case VPB_CID_DATE_TIME:
		len= strlen((char *)value);
	    if(strcmp((char *)value, "AUTO"))
		strcpy(cid->date_time, "AUTO");    
	    if(len == 8)
	    {
 	        // do valid check on numerics then copy if ok.
	        // valid chars are '0'-'9'
		valid=0;
		if ('0' == *((char*)value) )		//Month	
		    if(isinstr( *((char *)value+1),"0123456789"))	
			valid++;
		if (*((char *)value) == '1')
		    if (isinstr(*((char *)value+1),"012"))
			valid++;
		
		if (isinstr(*((char *)value+2),"0123"))	//Day
		    valid++;
		if (*((char*)value+2) == '3')
		{   
		    if (isinstr(*((char*)value+3),"01"))
			valid++;
		}	
		else
		{   
		    if (isinstr(*((char*)value+3),"0123456789"))
			valid++;	
		}
			
		if (isinstr(*((char*)value+4),"012"))	//HOUR
		    valid++;
		if (*((char *)value+4) == '2')
		{   
		    if (isinstr(*((char*)value+5),"0123"))
			valid++;
		}	
		else
		{   
		    if (isinstr(*((char*)value+5),"0123456789"))
		    	valid++;
		}	
				
		if (isinstr(*((char*)value+6),"012345"))	//Minute
		    valid++;
		if (isinstr(*((char*)value+7),"0123456789"))
		    valid++;
		
		// Now test if its ok
		if (valid ==7)
			strcpy(cid->date_time, (char *)value);
		else	
		    return -1;
		   
	    }
	    else
		return -1;	    
	    break;
	    
	case VPB_CID_CALLING_LINE_DN:
	    len=strlen((char*)value);
	    for(valid=x=0; x<len; x++)
		valid += isinstr(*(x+(char*)value),"0123456789 -");
	    if((len < 19) && valid == len)
	    {	    
		strcpy(cid->cldn, (char*)value);
	    	cid->ra_cldn[0] = 0;		// remove absence message
	    }	
	    else
	    {	//error
		return -1;
	    }	    
	    break;
	    
	case VPB_CID_CALLED_DN:
	    len=strlen((char*)value);
	    
	    for(valid=x=0; x<len; x++)
		valid += isinstr(*(x+(char*)value),"0123456789 -");
	    if((len < 19) && (valid == len))
		strcpy(cid->cdn, (char *)value);
	    else
	    {	//error
		return -1;
	    }
	    break;
	    
	case VPB_CID_RFA_CLDN:
	    if( *(char *)value == 'O')
	    {
		strcpy(cid->ra_cldn, "O");     // Number Unavailable
	    	cid->cldn[0] = 0;		// remove number
	    }	
	    else
	    {	    
		if( *(char *)value == 'P')
		{	
		    strcpy(cid->ra_cldn, "P");	// Number Withheld
	    	    cid->cldn[0] = 0;		// remove number
		}    
    		else
	       	    //error Invalid value
		    return -1;
	    }
	    break;
	    
	case VPB_CID_CALLER_NAME:
	    if(strlen((char *)value) < 21)
	    {
	        strcpy(cid->cn, (char *)value);
	    	cid->ra_cn[0] = 0;		// remove absence reason
	    }	
	    else
	    {	//error
		return -1;
	    }    
	    break;

	case VPB_CID_TEXT_PARAMETER:
	    if(strlen((char *)value) < 21)
	    {
	        strcpy(cid->cn, (char *)value);
	    	
	    }	
	    else
	    {	    //error
		    return -1;
	    }    
	    break;

	    
	    
	//  CID_RFA_CN	Reason For Absence of Calling Name    
	case VPB_CID_RFA_CN:
	    if( *(char *)value == 'O')
	    {
		strcpy(cid->ra_cn, "O");     // Number unavailable
	    	cid->cdn[0] = 0;		// remove number
	    }
	    else
		if( *(char *)value == 'P')
		{	
		    strcpy(cid->ra_cn, "P");	// Number with held
	    	    cid->cn[0] = 0;		// remove number
		}
    		else
		{    //error Invalid value
		    return -1;
		}    
	    break;
	    
	case VPB_CID_CALL_TYPE:
	    if (*(int *)value == VPB_CID_CT_VOICE)
		    cid->call_type= VPB_CID_CT_VOICE;

	    if (*(int *)value == VPB_CID_CT_RBWF) 	// "RING_BACK_WHEN_FREE"
		    cid->call_type= VPB_CID_CT_RBWF;
		    
	    if (*(int *)value == VPB_CID_CT_MW)	// "MESSAGE_WAITING"
		    cid->call_type= VPB_CID_CT_MW;
	    break;
	    
	case VPB_CID_NMSS:
	    
	    if(((*(int *)value) >0) && ((*(int *)value) <256))
	    	cid->nmss = *(int *)value;
	    else
	    {	// Error Invalid range
		return -1;
	    }	
	    break;
	    
	default:
	    return -1;   // ERROR
	    break;    
    }
  }
  else
	return -1;	  // ERROR 
  return 0;	
}


/*----------------------------------------------------------------------*\

	FUNCTION: vpb_cid_compose_dlp() 
	AUTHOR..: Peter Wintulich
	DATE....: Nov 2002
 
	Build the message dlp from the VPB_CID structure.
 
\*-----------------------------------------------------------------------*/

int WINAPI vpb_cid_compose_dlp(VPB_CID *message, char *dlp)
{
	int	ptr =0;			// pointer into dlp[n]
	int	csb =0;			// check sum byte accumulator
	int	x;

	time_t curtime;
        struct tm *loctime;
	char	temp[20];

	
	VPB_CID	*mesg;
	mesg	=(VPB_CID*) message;

	dlp[0]=0x80;	// Header byte;
	ptr = 2;	// leave space for message length byte
	
	// VPB_CID_CALL_TYPE
	if(mesg->call_type != 0)	// Always 3 bytes long
	{
	    dlp[ptr] = (char) VPB_CID_CALL_TYPE;	// label
	    dlp[ptr+1] = 1;			// size
	    dlp[ptr+2] =(char) mesg->call_type;	// data
	    ptr +=3;				// update dlp byte count
        }

	// VPB_CID_DATE_TIME
	if(mesg->date_time[0] != 0)		// Always 10 bytes
	{
	    dlp[ptr] = (char) VPB_CID_DATE_TIME;	// label
	    dlp[ptr+1] = 8;			// message length
	    if(strcmp(mesg->date_time, "AUTO") == 0)
	    {
		/* Get the current time. */
		curtime = time (NULL);
    		/* Convert it to local time representation. */
		loctime = localtime (&curtime);    
		sprintf(temp,"%.2d%.2d%.2d%.2d\n",(loctime->tm_mon)+1,
			       	loctime->tm_mday,
			       	loctime->tm_hour,
			       	loctime->tm_min);
		strcpy((&dlp[ptr])+2, temp);	
	    }
	    else
		strcpy((&dlp[ptr])+2, mesg->date_time);
	    
	    ptr += 10;
        }
	
	// VPB_CID_CALLING_LINE_DN	Calling Line Directory #
	if(mesg->cldn[0] != 0)			// Max 18 bytes
	{
	    dlp[ptr] = (char) VPB_CID_CALLING_LINE_DN;
	    dlp[ptr+1] = (char) strlen(mesg->cldn);
	    strcpy((&dlp[ptr])+2, mesg->cldn);
	    ptr += 2+strlen(mesg->cldn);
        }
	
	// VPB_CID_CALLED_DN	Called Directory #
	if(mesg->cdn[0] != 0)			// Max 18 bytes
	{
	    dlp[ptr] = (char) VPB_CID_CALLED_DN;
	    dlp[ptr+1] = (char) strlen(mesg->cdn);
	    strcpy((&dlp[ptr])+2, mesg->cdn);
	    ptr += 2+strlen(mesg->cdn);
        }
	
	// VPB_CID_RFA_CLDN	Reason For Absence of Calling Line Directory #
	if(mesg->ra_cldn[0] != 0)		// 3 bytes
	{
	    dlp[ptr] = (char) VPB_CID_RFA_CLDN;
	    dlp[ptr+1] = 1;
	    dlp[ptr+2] = (char) mesg->ra_cldn[0];
	    ptr += 3;
        }
	
	// VPB_CID_CALLER_NAME	Caller Name/Text Parameter
	if(mesg->cn[0] != 0)		// Max 20 ascii bytes
	{
	    dlp[ptr] = (char) VPB_CID_CALLER_NAME;
	    dlp[ptr+1] = (char) strlen(mesg->cn);
	    strcpy((&dlp[ptr])+2, mesg->cn);
	    ptr += 2+strlen(mesg->cn);
        }
                 	
	// VPB_CID_RFA_CN	Reason For Absence of Caller Name/Text Parameter
	if(mesg->ra_cn[0] != 0)	// 3 bytes
	{
	    dlp[ptr] = (char) VPB_CID_RFA_CN;
	    dlp[ptr+1] = 1;
	    dlp[ptr+2] = mesg->ra_cn[0];
	    ptr += 3;
        }
	
	// VPB_CID_NMSS		Network Message System Status 
	// (number of waiting messages in message bank)
	if(mesg->nmss != 0)		// 3 bytes
	{
	    dlp[ptr] = (char) VPB_CID_NMSS;
	    dlp[ptr+1] =  1;
	    dlp[ptr+2] = (char) mesg->nmss;
	    ptr += 3;
        }
	
	// Save Presentation layer message length
	dlp[1] = (char) ptr-2;
	
	// Now calculate & write checksum byte
	for(x=0; x < ptr; x++)
		csb += dlp[x];
	dlp[ptr]=(char) (0 - csb);
	ptr++;
	
	return ptr;
}


/*-----------------------------------------------------------------------\

	FUNCTION: vpb_cid_compose_wav()  
	AUTHOR..: Peter Wintulich
	DATE....: Nov 2002

	Encodes the Data Link Packet recived into a CP-FSK wave
	including CSS & MSS bit headers.  Returns *buff_wav containing
	*buff_cnt samples.

\-----------------------------------------------------------------------*/

int WINAPI vpb_cid_compose_wav(char *dlp, int dlp_len, short *buff_wav, 
			       int *buff_cnt)
	//	*dlp		Datalink Layer Packet
	//	dlp_len		*dlp length 
	//	*buff_wav	Pointer to start of Wave buffer
	//	*buf_cnt	Pointer to int, to return size of wave in buff_wav
{		 
	int	CSS = 300;	// CSS bit count
	int	MSS = 180;	// MSS bit count
	int	sc;		// sample count returned by f(n).
	int	i;
	int	accs = 0;	// Accumilated samples in buff_wav 
	
	//  These are required to keep phase and bit timing in sync.
	int	bit_time =0;	// used by bit_encode to track samples/bit
	double	sample_time =0; // used by bit_encode to track sinewave cycles

	for (i=0; i < CSS; i++)	// Compose CSS (alternate 1's & 0's)
	{
	    bit_encode((i & 1? 0 : 1), &sc, (short *)buff_wav+accs, &sample_time, &bit_time);
  	    accs +=sc;
	}
	
	for (i=0; i < MSS; i++)	// Compose MSS (all 1's)
	{
		bit_encode(1, &sc, &buff_wav[accs], &sample_time, &bit_time);
		accs +=sc;
	}
	
	for (i=0; i < dlp_len; i++)	// Compose data frame
		byte_encode(dlp[i], &accs, buff_wav, &sample_time, &bit_time);

	for (i=0; i < 5; i++)	// Trail some bits at the end (not required)
	{
		bit_encode(1, &sc, &buff_wav[accs], &sample_time, &bit_time);
		accs +=sc;
	}
	
	*buff_cnt = accs;	// Accumulated sample count
	return 0;
}	

/*-----------------------------------------------------------------------*\

	FUNCTION: byte_encode() 
	AUTHOR..: Peter Wintulich
	DATE....: Nov 2002
 
	Encode byte from DLPB into wave buffer *BUFF using bit_encode()
 
\*-----------------------------------------------------------------------*/

int byte_encode(char dlpb, int *scount, short *buff, double *stime, int *bt)
{	//	dlpb		Datalink Layer Packet Byte to encode
	//	*scount		Pointer for wave buffer current sample
	//	*buff		Pointer to head of wave buffer
	//	*stime		SINE time
	//	*bt		Bit Time

	int	i,p;
	int	sc;		// sample count per bit
	char	byte;		// temp char to shift bits out with

        p=*scount;
	bit_encode(0, &sc, &buff[*scount], stime, bt);	// Start Bit
	*scount +=sc;
	
	byte= dlpb;				// data bits 0-7
	for (i=0; i<8 ; i++)
	{	
		bit_encode((byte & 1? 1:0), &sc, &buff[*scount], stime, bt);
		*scount +=sc;
		byte = byte >> 1;
	}
	
	bit_encode(1, &sc, &buff[*scount], stime, bt);	// Stop Bit
	*scount +=sc;
	
	return 0;	
}	


/*-----------------------------------------------------------------------*\

	FUNCTION: bit_encode()
	AUTHOR..: Peter Wintulich
	DATE....: Nov 2002

        'Acc' & 'bt' need to be preserved by the cid composer (per
        thread) to maintain the phase relation ship and prevent bit
        width from accumilating any error, therefor causing a farming
        error.
 
        LAMP & HAMP are amplitudes for the sine generator,
        LAMP for mark, HAMP for space

\*-----------------------------------------------------------------------*/

int bit_encode(int bit,		// bit       	   Logic of bit to compose
        int *samples_count,	// *samples_count  returns number of samples
        short *samples, 	// *samples        array of short samples
        double *Acc, 		// *Acc		   Accumilation of sine Radians
        int *bt )		// *bt		   Bit time refrence
{

	int	x;
	int samples_bt[3] = {7,6,7};	// Number of sample times per bit
	

	for(x=0; x < samples_bt[(*bt)%3]; x++)	
	{
	    switch(COUNTRY)	// select tone set to use.
	    {
		case VPB_COUNTRY_UK:   
		    if(bit == 1)        //1300Hz
                    {
		        *Acc= *Acc + 1.021017612L;
		        samples[x] = (short) (LAMP * sin(*Acc));
		    }
                    else                //2100Hz
		    {
		        *Acc= *Acc + 1.649336143L;
		        samples[x] = (short) (HAMP * sin( *Acc));
		    }
		    break;
					
		case VPB_COUNTRY_USA:
	    	case VPB_COUNTRY_AU:	
	        default:	    
		    if(bit == 1) 	//1200Hz
		    {	    
			*Acc= *Acc + 0.942477769L;
			samples[x] = (short) (LAMP * sin( *Acc));
		    }	
		    else		//2200Hz
		    {
			*Acc= *Acc + 1.72787596L;	    
			samples[x] = (short) (HAMP * sin( *Acc));
		    }	
		    break;
	    }	    
	}
	*samples_count = samples_bt[*bt];
	(*bt)=((*bt)+1)%3;	// advance bit time
	
	return	0;
}


/*-----------------------------------------------------------------------*\

	FUNCTION: isinstr()
	AUTHOR..: Peter Wintulich
	DATE....: Nov 2002
 
 	Searches string list for char returns 1 if found

\*-----------------------------------------------------------------------*/

int	isinstr(char ch, char *list)
{
	unsigned	x=0;
	int		found=0;
	do
	{
		if( ch == *(list +x))
			found++;
	}while( ++x < strlen(list) && found == 0);	

	return found;
}	
