/*---------------------------------------------------------------------------*\
    FILE....: cid_jp.cpp
    TYPE....: WIN32 C++ Function
    AUTHOR..: John Kostogiannis
    DATE....: 14/11/97

    Japan DID demodulate and Decode functions

\*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*\

	Voicetronix Voice Processing Board (VPB) Software

	Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	Lesser General Public License for more details.
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	USA

\*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

				INCLUDES

\*-------------------------------------------------------------------------*/

#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <string.h>

#include "apifunc.h"
#include "cid_jp.h"
#include "vpbapi.h"
#include "wobbly.h"
#include "mess.h"

/*---------------------------------------------------------------------------*\

				DEFINES

\*---------------------------------------------------------------------------*/

	// parity control
#define PAR_8N	0
#define PAR_7E  1
#define PAR_7O  2

	// Decode frequencies for FSK bit stream
#define JP_F1	1300
#define JP_F2	2100
#define FS      8000    // sampling rate
#define NCOEF   7       // number of filter coefficients
#define NBASCII   8     // nunber of bits in each ASCII processing block
#define NAV     6       // number of samples to average energy over
#define MARKB   400     // number of sam. for MARK BITs header (50ms) 
#define NDCTMP  40      // number of sam. of CSS signal used for temp DC removal

#define MAX_JPRAW	(JP_RAW+10)

	// Header and Control bytes
#define	DLE	0x10
#define	SOH	0x01
#define	STX	0x02
#define	ETX	0x03
#define	SI	0x0f
#define	SO	0x0e

#define	H_BIN	0x07
#define	T_BIN	0x40

#define BIN_OTN 	0x02
#define DID_CND 	0x09
#define BIN_OTNRA 	0x04
/*	Byte +3 Reasons for not informing of the origenators Phone #
	'P'	0x50	// User Denial
	'O'	0x4f	// Unable to Offer Service
	'C'	0x43	// Origenated From Public Telephone
	'S'	0x53	// Service Conflict

*/
#define BIN_OTNEXP	0x21
#define DID_CNDEXP 	0x22
/*	Byte +3 Type of Number (TON) Expantion Signal 
	'0'	0x30	// Uncertain
	'1'	0x31	// International Number
	'2'	0x32	// Domestic Number
	'3'	0x33	// Network's Own Number
	'4'	0x34	// Local Number
	'6'	0x36	// Abbreviated Number
	'7'	0x37	// Resurved for extention
			// Others reserved

	bytes +4 & +5 Number Plan Identifier (NPI)
	'0','0'	0x30, 0x30	// Uncertain
	'0','1'	0x30, 0x31	// ISDN/telephone numbering plan E.164
	'0','3'	0x30, 0x33	// Data numbering plan X.121
	'0','4'	0x30, 0x34	// Telix numbering plan F.69
	'0','8'	0x30, 0x38	// Domestic numbering plan
	'0','9'	0x30, 0x39	// Private numbering plan
	'1','5'	0x31, 0x35	// resurved for extention
				// Others reserved
*/

/*-------------------------------------------------------------------------*\
 *
 *                                    STATICS
 *
\*-------------------------------------------------------------------------*/

//static FILE *fcd = NULL;

// FIR filter coefficients

static float coef[] = {
	(float)1.0000000e+000,
	(float)1.0515795e+000,
	(float)6.5481993e-001,
	(float)4.8815834e-001,
	(float)6.5481993e-001,
	(float)1.0515795e+000,
	(float)1.0000000e+000
};

// filter coeffs for averaging

static float avcoef[] = {
	(float)(1.0/6.0),
	(float)(1.0/6.0),
	(float)(1.0/6.0),
	(float)(1.0/6.0),
	(float)(1.0/6.0),
        (float)(1.0/6.0),
};


/*--------------------------------------------------------------------------*\

				      FUNCTION's

\*--------------------------------------------------------------------------*/

void demodulate_jp(float *demod, short *s, int n, float f1, float f2);

int extract_jp_pres( 
		char    	*pres,  	// char array for bytes
                int     	*npres, 	// byte count of pres
                float   	*demod, 	// bit streem data
                int     	n       	// word count of bit stream
                );

int bit_sync_ascii(int *bits_out, float *demod, int idx, int *valid);

int cid_jp_decode(
                char            *number,        // Callers number only
                VPB_CID_JP      *cid,           // JP Caller ID struct
                short           *in,            // Wave input
                int             n               // Sample count
                );

float filter(
                float in[],
                float coeff[],
                int   ncoeff
                );

int sgn(float x);

char bits2char(int bits[]);

int  crc(unsigned long *crc_acc, char data);

int byte_assemble(      float *demod,   // Demodulated buffer
                        int n,          // Sample count in demod
                        int *idx,       // Current index into demod
                        int parity,     // parity check + data mask option
                        char *byte,     // returned byte
			unsigned long	*cksumb);	// CRC WORD

int  bit_parity(int bits[]);

/*---------------------------------------------------------------------------*\
	FUNCTION: vpb_cid_jp_decode()
	AUTHOR..:
	DATE....:

\*---------------------------------------------------------------------------*/
int WINAPI vpb_cid_jp_decode(
		VPB_CID_JP	*cid,		// JP Caller ID struct
		short		*in,		// Wave input
		int		n		// Sample count
		)
{
	// Check if there is any big problems
	assert( cid != NULL );
	assert( in != NULL );
	assert( n > 1 );
	// OK lets go
	return (cid_jp_decode(NULL, cid, in, n));
}

/*---------------------------------------------------------------------------*\
	FUNCTION: cid_jp_decode()
	AUTHOR..: Peter Wintulich
	DATE....: 22 MAR 2004

	Takes Raw Audio sample and decodes to DID and BIN information
	Valid data if filled into VPB_JP_CID structure passed to the Fn.

	return()
		Return VPB_OK if sucess.
		Returns error code if fail to decode (use to identify point
		of fail in debug and test). Error will normaly indicate absence
		of DID and BIN information.

	error ==0	No error
	      > 0	error from extract_jp_pres()
      	      < 0	error from cid_jp_decode ()
\*---------------------------------------------------------------------------*/
int cid_jp_decode (
		char		*number,	// Callers number only
                VPB_CID_JP      *cid,           // JP Caller ID struct
                short           *in,            // Wave input
                int             n               // Sample count
                )
{
	char	pres[MAX_JPRAW];
	int	npres, bc=0;
	int	error = 0;
	int	i =0;
	int	state= 1;

	float *demod = new float[n]; 		// allocate space to decode
	assert( demod != NULL);			// OK ?

	// Demodulate audiosample to bit stream
	demodulate_jp(demod, in, n, JP_F1, JP_F2 );

	// display demodulated signal
	//for(i=0; i<(n/2); i++)
	//{
	//	printf("%1d,",sgn(demod[i]));
	//}	


	// Convert bit stream to byte stream of presentation layer only
	// and return byte count	
	error = extract_jp_pres(pres, &npres, demod, n); 
		//errors returned range 1..20 indicating point of fail
		
	// clean up temp varible, 
	delete demod;
	
	//printf(" \nerror= %d\n",error);
	//printf("npres=%d\n",npres);
 	//for(i=0; i<npres; i++)
	//         {
	//		printf("%2x,",pres[i]);
	//         }

	memcpy(cid->raw, pres, npres);
        cid->raw_size = npres;

	//	Empty any old data before proceeding
	cid->cnd[0] = 0;
	cid->cndexp[0] = 0;
	cid->otn[0] = 0;
	cid->otnra[0] = 0;
	cid->otnexp[0] = 0;
	
	//printf(" \nenter while\n");
	i=0;	

	state = 1;	// go till =0 or error
	while((state == 1) && (0 == error))
	{
	    	switch(pres[i])
	        {
		    case DID_CND:	// Called Number Data 
			if((bc= pres[i+1]) < OTN_L )
			{
			    memcpy(cid->cnd, &pres[i+2], bc);
			    cid->cnd[bc] = 0; // Null Terminate
			}
			else
			{
			    cid->cnd[0] = 0; // Zero length string
			    error = -2;
			}
			break;

		    case DID_CNDEXP:	// Called Number Data Expansion
			if((bc=pres[i+1]) == 3)
			{
			    memcpy(cid->cndexp, &pres[i+2], bc);
			    cid->cndexp[bc] = 0; // Null Terminate
			}
			else
			{
			    cid->cndexp[0] = 0;	// Zero length string
			    error = -3;		// Inconsistent	
			}    
			break;

		    case BIN_OTN:	// Origenators Tel. Number
			if((bc=pres[i+1]) <= 20)
			{
			    memcpy(cid->otn, &pres[i+2], bc);
			    cid->otn[bc] = 0;	// Null Term.
			}	
			break;

		    case BIN_OTNRA:	// OTN Reason for Absence
			if((bc=pres[i+1]) == 1)
			{
			    memcpy(cid->otnra, &pres[i+2], bc);
			    cid->otnra[bc] = 0;	// Null Term.
			}
			else
			{
			    cid->otnra[0] = 0;	// Zero length string
			    error = -4;		// Inconsistent	
			}
			break;

		    case BIN_OTNEXP:	// OTN Expansion Signal
			if((bc=pres[i+1]) == 3)
			{
			    memcpy(cid->otnexp, &pres[i+2], pres[i+1]);
			    cid->otnexp[(int)(pres[i+1])] = 0;	// Null Term.
			}
			else
			{
			    cid->otnexp[0] = 0;	// Zero length string
			    error = -5;		// Inconsistent	
			}    
			break;
	        } // end: switch(pres[i])
	        i += bc;
	        if( npres <= i)
			state = 0;	// we are done
	}	// end: while((state == 1) && (0 == error))

	if(error != 0)
		return(error);
	else
		return(VPB_OK);	// VPB_OK = No error if all ok
}

/*---------------------------------------------------------------------------*\
	FUNCTION: 	extract_jp_pres()
	AUTHOR..:	Peter Wintulich
	DATE....:	22/APR/2004

	Convert Demodulated stream to bytes, Verify header, 

	Error numbers 1..14 used, resurved to 20
	0 == no error (VPB_OK)

	
\*---------------------------------------------------------------------------*/
int extract_jp_pres(	char 	*pres,	// char array for bytes
			int	*npres, // byte count of pres
			float	*demod,  // bit streem data
			int	n	// Nuimber of samples in bit stream demod[]
			)
{

	#define	SMARK	2
	#define VMARK	3
	#define SSTART	4
	#define	EXITS	10
	#define	S2	12
	#define	S3	13
	#define	S4	14
	#define	EPL	15
	
	int	idx;
	int	error =0; 
	int	state;
	int	i,j,k;
	int	tos, tos_l;
	char	crc_l, crc_h;
	unsigned long	cksum;
	int	dle_trail=0;
	char	ascii_char;
        float           low=0.0, hi=0.0, dc=0.0;
        int             lc=0, hc=0;
	
	cksum = 0;
	
	mprintf("extract_jp_pres(%x,%x,%x,%d)\n",pres, npres, demod, n);
	*npres = 0;
	idx = 0;
	state=SMARK;
	do{
	    switch(state)	
	    {
		case	SMARK:  // Search for beginning of MARK
			mprintf("SMARK idx=%d\n",idx);       	
			i=idx;
			while((sgn(demod[i]) ==0) && (state == EXITS))
			{
				printf("%1d,",sgn(demod[i]));
				i++;	
				if(i >= n)		// More samples ? 
				{
					error=1;
					state=EXITS;
					mprintf("(SMARK) i=%d\n",i);
				}
			}
			
			if(error == 0)			// If no error then
			{
				idx=i;		// save current place were upto
				i=0;		// reset 
				state=VMARK;	// now measer mark len
			}
			break;

		case	VMARK:	// Measer (verify) Mark lead in.
			mprintf("VMARK idx=%d, i=%d, error=%d\n",idx,i,error);
			while((state == VMARK) && ((i+idx) < n))
			{
			    if(sgn(demod[i+idx]) ==1) 
			    	i++;	
			    else 
			    {
				if(i > MARKB)
				{	// Test if there are more than 3 zero's
					j=0;
					for(k=0; k<6; k++)
	                                	j +=sgn(demod[k+idx+i]);
					if(j < 3)
					{
						state = SSTART;
						idx += (i-1);	// leave 1 for zcross det
					}
					else
						i++;
				}
				else
				{	
					idx +=i+1;
					i=0;
				}
			    }
			    
			    if((i+idx) >= n)
			    {
				state=EXITS;
				error=2;
			    }
			}	
			break;

		case	SSTART:	// dle
			mprintf("DLE idx=%d\n",idx);
			// First do the DC offset average over next 1000 samples
			low=0.0;// Accumilated LOW samples
			hi=0.0;	// Accumilated HIGH samples
			dc=0.0;	// Final DC shift to apply
			lc=0;	// count of LOW (neg.) sig. samples
			hc=0;	// count of HIGH (pos.) sig. samples
		        for(i=idx; i<(idx+1000); i++)
		        {	// only calculate with samples of strong signal
				if(demod[i] > 100000.0)
				{
					hi+= demod[i];
					hc++;
				}
				if(demod[i] < -100000.0)
				{
					low+= demod[i];
					lc++;
				}
			}
			mprintf("hi=%f, low=%f\n",hi/(float)hc, low/(float)lc);
			dc= (low/(float)lc)+(((hi/(float)hc)-(low/(float)lc))/2);
			mprintf("DC=%f\n", dc);
			// don't apply offset yet, first check that it is realy data we
			// are reading and not noise from off hook physical event.
			
			// now try reading data
			if(byte_assemble(demod, n, &idx, PAR_7E, &ascii_char, &cksum) !=0)
			{
				if(ascii_char == DLE)	// If good apply offset & continue
		       		{
					state = S2;		
					// correct DC offset across entire signal
					for(i=0; i<n; i++) 
						demod[i] -= dc;	 
				}
				else
				{
				        //error= 3;
					cksum =0;
					state=SMARK;
				}
			}
			else
			{
				if(idx < n)
				{
					cksum =0;
					state=SMARK;
				}
				else
				{
					state=EXITS;
				        error= 4;
				}
			}
			break;

		case	S2:	// soh
			mprintf("SOH idx=%d\n",idx);
			if(byte_assemble(demod, n, &idx, PAR_7E, &ascii_char, &cksum) !=0)
			{
				if(ascii_char == SOH)
				{
					state= S3;
				}
				else
					error= 5;
			}
			else
                        {
                                state=EXITS;
                                error= 6;
                        }
			break;

		case	S3:	// HEADER NOTE: CRC starts from here
			mprintf("HEADER idx=%d\n",idx);
			cksum =0;	// Resst CRC accumilator, first valid byte.

			if(byte_assemble(demod, n, &idx, PAR_7E, &ascii_char, &cksum) !=0)
			{	
				if(ascii_char == H_BIN)
				{
					state= S4;		
				}
				else
					error= 7;
			}
			else
			{
				state=EXITS;
				error= 8;
			}	
			break;

		case	S4:	// DLE & STX
			mprintf("S4 idx=%d\n",idx);
			byte_assemble(demod, n, &idx, PAR_7E, &ascii_char, &cksum);
				
			if(ascii_char == DLE)
			{
				dle_trail=1;			
				byte_assemble(demod, n, &idx, PAR_7E, &ascii_char, &cksum);
				
			}
			if(ascii_char == STX)
			{
				state= EPL;
			}
			else
				error= 9;
			break;

		case	EPL:	// Extract presentation layer
			mprintf("EPL idx=%d\n",idx);
			
			// Get Type Of Service
			if(byte_assemble(demod, n, &idx, PAR_7E, &ascii_char, &cksum) ==0)
			{
				state= EXITS;
				error= 10;		// bad parity
			}
			else
			{	
				tos= (int) ascii_char;
				//printf("tos=%2d\n",tos);
			}
			
			// Get Length
			if((error ==0)&&(byte_assemble(demod, n, &idx, PAR_7E, &ascii_char, &cksum) ==0))
			{
			        state= EXITS;
                                error= 11;               // bad parity
                        }
			else
			{
			        tos_l= (int) ascii_char;
				//printf("tos_l=%2d\n",tos_l);
			}

			// read presentation layer into buffer
			*npres = tos_l;
			for(i=0; i<tos_l; i++)
			{
				if(byte_assemble(demod, n, &idx, PAR_7E, &ascii_char, &cksum) !=0)
				{
	                        	pres[i]= ascii_char;
					//printf("pres[%d]=%d,'%c'\n",i,pres[i],pres[i]);
				}
				else
				{
				        state= EXITS;
					error= 12;               // bad parity
				}
				
			}
			if((error == 0) &&(dle_trail == 1))
			{
				if(byte_assemble(demod, n, &idx, PAR_7E, &ascii_char, &cksum) !=0)
				{
					if(ascii_char == DLE)
					{
						mprintf("DLE,");
					}
				}
				else
				{
				        state= EXITS;
				        error= 13;               // bad Parity on DLE
				}
			}
			
			// check for ETX  NOTE: this is the last byte to the CRC 
			if((error ==0)&&(byte_assemble(demod, n, &idx, PAR_7E, &ascii_char, &cksum) !=0))
			{
				if(ascii_char == ETX)
				{
					mprintf("ETX\n");
				}	
				else
					error =14;	       // NOT ETX
			}
			
			if(error ==0)	// if ok then read CRC bytes note 8 bit data
			{

				byte_assemble(demod, n, &idx, PAR_8N, &crc_h, &cksum);
				byte_assemble(demod, n, &idx, PAR_8N, &crc_l, &cksum);
				mprintf("crc16 = %8x\n",cksum);
				if(cksum != 0)
					error = 15;		// bad crc calculation
			}
			
			state= EXITS;
			break;	
			
		default:
			state= EXITS;
			break;	
	    }		
	    if(error !=0)
		    state= EXITS;

	}while(state != EXITS);

	return(error);
}

/*---------------------------------------------------------------------------*\
	FUNCTION:	byte_assemble() 
	AUTHOR..:	Peter Wintulich
	DATE....:	20/APR/2004

	Assemble byte from demodulated string & current pointer,
       	Apply Parity check. update pointer.

\*---------------------------------------------------------------------------*/
int byte_assemble(	float *demod,	// Demodulated buffer
	       		int n,		// Sample count in demod
		        int *idx, 	// Current index into demod
			int parity,	// parity check + data mask option
			char *byte,	// returned byte
			unsigned long	*cksumb)	// Checksum word
{
	int	valid =1;
	int	bits_out[10];
	int	i;
	int	dc=0;
	char	chr;
	
	*idx = bit_sync_ascii(bits_out, demod, *idx, &valid);	// demod to data bits
	
	// Debug assist
	mprintf("         |          0      1      2      3      4     5      6      7\n");
	mprintf("IDX=%05d:",*idx);	
	for(i=0; i<100; i++)
	{
		mprintf("%c",(sgn(dc + demod[-63 + i+ *idx]) == 1 ?'1':'0'));
	}
	
	if(valid != 0)
	{
	    chr = (char)  bits2char(bits_out);	
	    switch(parity)
	    {
		case 	PAR_7O:		// 7 databit odd parity
			if(bit_parity(bits_out) != 1)
				valid = 0;	// Not valid
			*byte = chr & 0x7f;	// Remove Parity bit
			crc(cksumb, chr);      // Do CRC all 8 bits
			break;

		case 	PAR_7E:         // 7 databit even parity
			if(bit_parity(bits_out) != 0)
				valid = 0;	// Not valid
			*byte = chr & 0x007f;	// Remove Parity bit 
			crc(cksumb, chr); 	// Do CRC all 8 bits
			break;

		case	PAR_8N:		// 8 databit no patity
			*byte = chr;
			crc(cksumb, chr);	// Do CRC
			break;
		default:
			valid = 0;      // Not valid	
	    }
	}
	mprintf("=%02x,%01d\n",*byte,valid);
	return (valid);
}

/*---------------------------------------------------------------------------*\
	FUNCTION:	crc(crc_acc, data) 
	AUTHOR..:	Peter Wintulich
	DATE....:	9 JULY 2004

	CRC16 checking function 
	Need X16+X12+X5+1. Bits enter CRC checker in the order they are recived
	from the raw data streem. So 7 bits and the parity bit. crc_acc should be
	loaded with 0 before start. 

\*---------------------------------------------------------------------------*/
int	crc(unsigned long *crc_acc, char data )
{
	int	i, bit;
	unsigned long	genpoly= 0x00011021;
	// 		 bin 0001 0001 0000 0010 0001 = (2^16+2^12+2^5+2^0)
	
	for(i=0; i<8; i++)	// go from bit 
	{
		bit=(data & (1<<i) ?1:0);	// sample bit
		*crc_acc= (*crc_acc <<1) + bit; // shift bit into CRC acc
		if((*crc_acc & (1<<16))!=0)	// is overflow 1 from 16th bit
		{	
			*crc_acc= (*crc_acc) ^ genpoly;	// xor
		}
	}
	
	return((*crc_acc) & 0xffff);
}

/*---------------------------------------------------------------------------*\
	FUNCTION:	bit_parity(int bits[]) 
	AUTHOR..:	Peter Wintulich
	DATE....:	22/APR/2004

	check parity of a bits stored in char array.

\*---------------------------------------------------------------------------*/
int	bit_parity(int bits[])
{
	int 	i;
	int	j = 0;

	for(i=0; i<8; i++)
	{
		j+= bits[i];
	}
	
	return(j & 0x0001);
}

/*---------------------------------------------------------------------------*\
	FUNCTION: 	demodulate_jp()
	AUTHOR..:	David Rowe + 
	DATE....:	

	Demodulate FSK encoded signal.

\*---------------------------------------------------------------------------*/
void demodulate_jp(
	   float *demod,		// output demodlated samples	
	   short *s,			// input samples
	   int   n,			// length of s and demod
	   float f1,			// low freq. fsk
	   float f2			// high freq. fsk
	)
{
	int   i,j;
	float pi = (float)4.0*(float)atan((float)1.0);
	float w1 = 2*pi*f1/FS;
	float w2 = 2*pi*f2/FS;
	float x_sinF1[NCOEF], x_cosF1[NCOEF], x_sinF2[NCOEF], x_cosF2[NCOEF];
	float xlpsf1, xlpcf1, xlpsf2, xlpcf2;
	float ef1[NAV], ef2[NAV];
	float avef1, avef2;

	//printf("demodulate_jp(%x,%x,%d,%f,%f)\n",demod, s, n, f1, f2 );
	// initialise memories

	for(i=0; i<NCOEF; i++) {
		x_sinF1[i] = (float)0.0;
		x_cosF1[i] = (float)0.0;
		x_sinF2[i] = (float)0.0;
		x_cosF2[i] = (float)0.0;
	}

	for(i=0; i<NAV; i++) {
		ef1[i] = (float)0.0;
		ef2[i] = (float)0.0;
	}

	for(i=0; i<n; i++) {
		// subject signal to mixers
		x_sinF1[NCOEF-1] = s[i]*(float)sin(w1*i);
		x_cosF1[NCOEF-1] = s[i]*(float)cos(w1*i);
		x_sinF2[NCOEF-1] = s[i]*(float)sin(w2*i);
		x_cosF2[NCOEF-1] = s[i]*(float)cos(w2*i);

		// low pass filter
		xlpsf1 = filter(&x_sinF1[NCOEF-1], coef, NCOEF);
		xlpcf1 = filter(&x_cosF1[NCOEF-1], coef, NCOEF);
		xlpsf2 = filter(&x_sinF2[NCOEF-1], coef, NCOEF);
		xlpcf2 = filter(&x_cosF2[NCOEF-1], coef, NCOEF);

		// sum the energies for each arm
		ef1[NAV-1] = xlpsf1*xlpsf1 + xlpcf1*xlpcf1;
		ef2[NAV-1] = xlpsf2*xlpsf2 + xlpcf2*xlpcf2;

		// average energies over past 6 samples
		avef1 = filter(&ef1[NAV-1], avcoef, NAV);
		avef2 = filter(&ef2[NAV-1], avcoef, NAV);

		// output is difference of upper and lower arms
		demod[i] = avef1 - avef2;

		// update memories
		for(j=0; j<NCOEF-1; j++) {
			x_sinF1[j] = x_sinF1[j+1];
			x_cosF1[j] = x_cosF1[j+1];
			x_sinF2[j] = x_sinF2[j+1];
			x_cosF2[j] = x_cosF2[j+1];
		}
		for(j=0; j<NAV-1; j++) {
			ef1[j] = ef1[j+1];
			ef2[j] = ef2[j+1];
		}
	}

}

